<?php
/**
 * Delcampe Stamp Profile Fields Class
 * 
 * Provides stamp-specific field rendering for profiles including shipping model integration.
 * Handles UI generation for stamp attributes, shipping, payment, and description settings.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Profiles
 * @since 1.3.0.0
 * @version 1.3.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Include the stamp profile fields helper
require_once plugin_dir_path( __FILE__ ) . 'stamp-profile-fields.php';

/**
 * Class Delcampe_Stamp_Profile_Fields
 * 
 * Renders stamp-specific fields in profile editor
 * 
 * @version 1.3.0.0
 */
class Delcampe_Stamp_Profile_Fields {
    
    /**
     * Shipping models instance
     * @var Delcampe_Shipping_Models
     * @since 1.3.0.0
     */
    private $shipping_models;
    
    /**
     * Constructor
     * 
     * @since 1.3.0.0
     */
    public function __construct() {
        // Get shipping models instance if available
        if ( class_exists( 'Delcampe_Shipping_Models' ) ) {
            require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-delcampe-shipping-models.php';
            $this->shipping_models = Delcampe_Shipping_Models::get_instance();
        }
    }
    
    /**
     * Render stamp-specific fields
     * 
     * @param array $profile Profile data
     * @return string HTML output
     */
    public function render_stamp_fields( $profile ) {
        $stamp_settings = isset( $profile['stamp_settings'] ) ? $profile['stamp_settings'] : array();
        
        ob_start();
        ?>
        <div class="postbox">
            <h2 class="hndle"><?php _e( 'Stamp-Specific Settings', 'wc-delcampe-integration' ); ?></h2>
            <div class="inside">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="stamp_condition">
                                <?php _e( 'Default Condition', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="stamp_condition" name="stamp_settings[default_condition]">
                                <?php foreach ( delcampe_get_stamp_conditions_translated() as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" 
                                            <?php selected( isset( $stamp_settings['default_condition'] ) ? $stamp_settings['default_condition'] : 'Used', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="catalog_prefix">
                                <?php _e( 'Catalog System', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="catalog_prefix" name="stamp_settings[catalog_prefix]">
                                <?php foreach ( delcampe_get_catalog_systems_translated() as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" 
                                            <?php selected( isset( $stamp_settings['catalog_prefix'] ) ? $stamp_settings['catalog_prefix'] : 'Michel', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php _e( 'Include in Description', 'wc-delcampe-integration' ); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="checkbox" 
                                           name="stamp_settings[add_catalog_number]" 
                                           value="1" 
                                           <?php checked( isset( $stamp_settings['add_catalog_number'] ) ? $stamp_settings['add_catalog_number'] : '1', '1' ); ?>>
                                    <?php _e( 'Add catalog number to title', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="stamp_settings[perforation_details]" 
                                           value="1" 
                                           <?php checked( isset( $stamp_settings['perforation_details'] ) ? $stamp_settings['perforation_details'] : '1', '1' ); ?>>
                                    <?php _e( 'Include perforation details', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="stamp_settings[watermark_info]" 
                                           value="1" 
                                           <?php checked( isset( $stamp_settings['watermark_info'] ) ? $stamp_settings['watermark_info'] : '1', '1' ); ?>>
                                    <?php _e( 'Include watermark information', 'wc-delcampe-integration' ); ?>
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="gum_condition">
                                <?php _e( 'Gum Condition', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="gum_condition" name="stamp_settings[gum_condition]">
                                <?php foreach ( delcampe_get_gum_conditions_translated() as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" 
                                            <?php selected( isset( $stamp_settings['gum_condition'] ) ? $stamp_settings['gum_condition'] : '', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="centering">
                                <?php _e( 'Centering', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="centering" name="stamp_settings[centering]">
                                <?php foreach ( delcampe_get_centering_options_translated() as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" 
                                            <?php selected( isset( $stamp_settings['centering'] ) ? $stamp_settings['centering'] : '', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="grade">
                                <?php _e( 'Grade', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="grade" name="stamp_settings[grade]">
                                <?php foreach ( delcampe_get_grade_options_translated() as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" 
                                            <?php selected( isset( $stamp_settings['grade'] ) ? $stamp_settings['grade'] : '', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php _e( 'Certification', 'wc-delcampe-integration' ); ?>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" 
                                       id="certificate" 
                                       name="stamp_settings[certificate]" 
                                       value="1" 
                                       <?php checked( isset( $stamp_settings['certificate'] ) ? $stamp_settings['certificate'] : '0', '1' ); ?>>
                                <?php _e( 'Has certificate', 'wc-delcampe-integration' ); ?>
                            </label>
                            <div id="certificate-details-wrapper" style="margin-top: 10px; <?php echo ( isset( $stamp_settings['certificate'] ) && $stamp_settings['certificate'] == '1' ) ? '' : 'display:none;'; ?>">
                                <label>
                                    <?php _e( 'Certificate Details:', 'wc-delcampe-integration' ); ?>
                                    <input type="text" 
                                           name="stamp_settings[certificate_details]" 
                                           value="<?php echo esc_attr( isset( $stamp_settings['certificate_details'] ) ? $stamp_settings['certificate_details'] : '' ); ?>" 
                                           class="regular-text">
                                </label>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php _e( 'Expertization', 'wc-delcampe-integration' ); ?>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" 
                                       id="expertized" 
                                       name="stamp_settings[expertized]" 
                                       value="1" 
                                       <?php checked( isset( $stamp_settings['expertized'] ) ? $stamp_settings['expertized'] : '0', '1' ); ?>>
                                <?php _e( 'Has been expertized', 'wc-delcampe-integration' ); ?>
                            </label>
                            <div id="expertizer-details-wrapper" style="margin-top: 10px; <?php echo ( isset( $stamp_settings['expertized'] ) && $stamp_settings['expertized'] == '1' ) ? '' : 'display:none;'; ?>">
                                <label>
                                    <?php _e( 'Expertizer Name:', 'wc-delcampe-integration' ); ?>
                                    <input type="text" 
                                           name="stamp_settings[expertizer_name]" 
                                           value="<?php echo esc_attr( isset( $stamp_settings['expertizer_name'] ) ? $stamp_settings['expertizer_name'] : '' ); ?>" 
                                           class="regular-text">
                                </label>
                            </div>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Toggle certificate details
            $('#certificate').on('change', function() {
                if ($(this).is(':checked')) {
                    $('#certificate-details-wrapper').slideDown();
                } else {
                    $('#certificate-details-wrapper').slideUp();
                }
            });
            
            // Toggle expertizer details
            $('#expertized').on('change', function() {
                if ($(this).is(':checked')) {
                    $('#expertizer-details-wrapper').slideDown();
                } else {
                    $('#expertizer-details-wrapper').slideUp();
                }
            });
        });
        </script>
        <?php
        
        return ob_get_clean();
    }
    
    /**
     * Render shipping fields with Delcampe shipping model integration
     * 
     * @param array $profile Profile data
     * @return string HTML output
     * @since 1.3.0.0
     */
    public function render_shipping_fields( $profile ) {
        $shipping_settings = isset( $profile['shipping_settings'] ) ? $profile['shipping_settings'] : array();
        $shipping_model_id = isset( $profile['shipping_model_id'] ) ? $profile['shipping_model_id'] : 0;
        
        // Get available shipping models
        $shipping_models = array();
        if ( $this->shipping_models ) {
            $models = $this->shipping_models->get_shipping_models();
            if ( ! is_wp_error( $models ) ) {
                $shipping_models = $models;
                // Debug log the shipping models
                delcampe_log('[Delcampe Stamp Profile Fields] Shipping models: ' . print_r($shipping_models, true));
            } else {
                delcampe_log('[Delcampe Stamp Profile Fields] Error getting shipping models: ' . $models->get_error_message());
            }
        } else {
            delcampe_log('[Delcampe Stamp Profile Fields] No shipping models manager instance');
        }
        
        ob_start();
        ?>
        <div class="postbox">
            <h2 class="hndle"><?php _e( 'Shipping Settings', 'wc-delcampe-integration' ); ?></h2>
            <div class="inside">
                <table class="form-table">
                    <?php if ( ! empty( $shipping_models ) ) : ?>
                        <tr>
                            <th scope="row">
                                <label for="shipping_model_id">
                                    <?php _e( 'Delcampe Shipping Model', 'wc-delcampe-integration' ); ?>
                                    <span class="required">*</span>
                                </label>
                            </th>
                            <td>
                                <select id="shipping_model_id" name="shipping_model_id" required>
                                    <option value=""><?php echo esc_html__( '— Select Shipping Model —', 'wc-delcampe-integration' ); ?></option>
                                    <?php foreach ( $shipping_models as $model ) : ?>
                                        <option value="<?php echo esc_attr( $model['id'] ); ?>" 
                                                <?php selected( $shipping_model_id, $model['id'] ); ?>>
                                            <?php echo esc_html( $model['name'] ); ?>
                                            <?php if ( ! empty( $model['description'] ) ) : ?>
                                                - <?php echo esc_html( $model['description'] ); ?>
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description">
                                    <?php 
                                    printf(
                                        __( 'Select a shipping model from your Delcampe account. Configure shipping models in %sDelcampe Settings%s.', 'wc-delcampe-integration' ),
                                        '<a href="' . admin_url( 'admin.php?page=delcampe-settings&tab=shipping' ) . '">',
                                        '</a>'
                                    ); 
                                    ?>
                                </p>
                                
                                <?php if ( $shipping_model_id > 0 ) : ?>
                                    <?php
                                    // Find selected model details
                                    $selected_model = null;
                                    foreach ( $shipping_models as $model ) {
                                        if ( $model['id'] == $shipping_model_id ) {
                                            $selected_model = $model;
                                            break;
                                        }
                                    }
                                    ?>
                                    
                                    <?php if ( $selected_model && ! empty( $selected_model['costs'] ) ) : ?>
                                        <div style="margin-top: 15px; padding: 10px; background: #f0f0f0; border-radius: 3px;">
                                            <strong><?php _e( 'Selected Model Details:', 'wc-delcampe-integration' ); ?></strong>
                                            <ul style="margin: 5px 0;">
                                                <?php foreach ( $selected_model['costs'] as $cost ) : ?>
                                                    <li>
                                                        <?php
                                                        $zone_text = $cost['country'] ? $cost['country'] : __( 'Default', 'wc-delcampe-integration' );
                                                        if ( $cost['zone'] ) {
                                                            $zone_text .= ' - ' . $cost['zone'];
                                                        }
                                                        echo esc_html( $zone_text ) . ': ';
                                                        echo wc_price( $cost['price'] );
                                                        
                                                        if ( $cost['additional_price'] > 0 ) {
                                                            echo ' (+' . wc_price( $cost['additional_price'] ) . ' ' . __( 'per additional item', 'wc-delcampe-integration' ) . ')';
                                                        }
                                                        ?>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        </div>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php else : ?>
                        <tr>
                            <td colspan="2">
                                <div class="notice notice-warning inline">
                                    <p>
                                        <?php 
                                        printf(
                                            __( 'No shipping models found. Please configure shipping models in your Delcampe account and then %srefresh the models%s.', 'wc-delcampe-integration' ),
                                            '<a href="' . admin_url( 'admin.php?page=delcampe-settings&tab=shipping' ) . '">',
                                            '</a>'
                                        ); 
                                        ?>
                                    </p>
                                </div>
                            </td>
                        </tr>
                    <?php endif; ?>
                    
                    <tr>
                        <th scope="row">
                            <?php _e( 'Additional Shipping Options', 'wc-delcampe-integration' ); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="checkbox" 
                                           name="shipping_settings[combined_shipping]" 
                                           value="1" 
                                           <?php checked( isset( $shipping_settings['combined_shipping'] ) ? $shipping_settings['combined_shipping'] : '1', '1' ); ?>>
                                    <?php _e( 'Offer combined shipping for multiple items', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="shipping_settings[insurance_optional]" 
                                           value="1" 
                                           <?php checked( isset( $shipping_settings['insurance_optional'] ) ? $shipping_settings['insurance_optional'] : '1', '1' ); ?>>
                                    <?php _e( 'Offer optional shipping insurance', 'wc-delcampe-integration' ); ?>
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="handling_time">
                                <?php _e( 'Handling Time', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <input type="number" 
                                   id="handling_time" 
                                   name="shipping_settings[handling_time]" 
                                   value="<?php echo esc_attr( isset( $shipping_settings['handling_time'] ) ? $shipping_settings['handling_time'] : '2' ); ?>" 
                                   min="1" 
                                   max="30" 
                                   class="small-text">
                            <?php _e( 'business days', 'wc-delcampe-integration' ); ?>
                            <p class="description">
                                <?php _e( 'Time needed to prepare and ship the item after payment is received.', 'wc-delcampe-integration' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="return_policy">
                                <?php _e( 'Return Policy', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="return_policy" name="shipping_settings[return_policy]">
                                <option value="0" <?php selected( isset( $shipping_settings['return_policy'] ) ? $shipping_settings['return_policy'] : '14', '0' ); ?>>
                                    <?php _e( 'No returns accepted', 'wc-delcampe-integration' ); ?>
                                </option>
                                <option value="7" <?php selected( isset( $shipping_settings['return_policy'] ) ? $shipping_settings['return_policy'] : '14', '7' ); ?>>
                                    <?php _e( '7 days return', 'wc-delcampe-integration' ); ?>
                                </option>
                                <option value="14" <?php selected( isset( $shipping_settings['return_policy'] ) ? $shipping_settings['return_policy'] : '14', '14' ); ?>>
                                    <?php _e( '14 days return', 'wc-delcampe-integration' ); ?>
                                </option>
                                <option value="30" <?php selected( isset( $shipping_settings['return_policy'] ) ? $shipping_settings['return_policy'] : '14', '30' ); ?>>
                                    <?php _e( '30 days return', 'wc-delcampe-integration' ); ?>
                                </option>
                            </select>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        <?php
        
        return ob_get_clean();
    }
    
    /**
     * Render payment fields
     * 
     * @param array $profile Profile data
     * @return string HTML output
     */
    public function render_payment_fields( $profile ) {
        $payment_settings = isset( $profile['payment_settings'] ) ? $profile['payment_settings'] : array();
        $selected_methods = isset( $payment_settings['payment_methods'] ) ? $payment_settings['payment_methods'] : array( 'PayPal', 'BankTransfer' );
        
        ob_start();
        ?>
        <div class="postbox">
            <h2 class="hndle"><?php _e( 'Payment Settings', 'wc-delcampe-integration' ); ?></h2>
            <div class="inside">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <?php _e( 'Accepted Payment Methods', 'wc-delcampe-integration' ); ?>
                        </th>
                        <td>
                            <fieldset>
                                <?php foreach ( delcampe_get_payment_methods_translated() as $value => $label ) : ?>
                                    <label>
                                        <input type="checkbox" 
                                               name="payment_settings[payment_methods][]" 
                                               value="<?php echo esc_attr( $value ); ?>" 
                                               <?php checked( in_array( $value, $selected_methods ) ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </label><br>
                                <?php endforeach; ?>
                            </fieldset>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="payment_deadline">
                                <?php _e( 'Payment Deadline', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <input type="number" 
                                   id="payment_deadline" 
                                   name="payment_settings[payment_deadline]" 
                                   value="<?php echo esc_attr( isset( $payment_settings['payment_deadline'] ) ? $payment_settings['payment_deadline'] : '7' ); ?>" 
                                   min="1" 
                                   max="30" 
                                   class="small-text">
                            <?php _e( 'days after auction end', 'wc-delcampe-integration' ); ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="payment_instructions">
                                <?php _e( 'Payment Instructions', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <textarea id="payment_instructions" 
                                      name="payment_settings[payment_instructions]" 
                                      rows="3" 
                                      class="large-text"><?php echo esc_textarea( isset( $payment_settings['payment_instructions'] ) ? $payment_settings['payment_instructions'] : '' ); ?></textarea>
                            <p class="description">
                                <?php _e( 'Additional payment instructions for buyers (optional).', 'wc-delcampe-integration' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="paypal_email">
                                <?php _e( 'PayPal Email', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <input type="email" 
                                   id="paypal_email" 
                                   name="payment_settings[paypal_email]" 
                                   value="<?php echo esc_attr( isset( $payment_settings['paypal_email'] ) ? $payment_settings['paypal_email'] : '' ); ?>" 
                                   class="regular-text">
                            <p class="description">
                                <?php _e( 'Your PayPal email address (if accepting PayPal).', 'wc-delcampe-integration' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        <?php
        
        return ob_get_clean();
    }
    
    /**
     * Render description template fields
     * 
     * @param array $profile Profile data
     * @return string HTML output
     */
    public function render_description_fields( $profile ) {
        $description_settings = isset( $profile['description_settings'] ) ? $profile['description_settings'] : array();
        
        ob_start();
        ?>
        <div class="postbox">
            <h2 class="hndle"><?php _e( 'Description Template Settings', 'wc-delcampe-integration' ); ?></h2>
            <div class="inside">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="template_type">
                                <?php _e( 'Template Type', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="template_type" name="description_settings[template_type]">
                                <?php foreach ( delcampe_get_description_templates_translated() as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" 
                                            <?php selected( isset( $description_settings['template_type'] ) ? $description_settings['template_type'] : 'stamp_standard', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php _e( 'Include in Description', 'wc-delcampe-integration' ); ?>
                        </th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="checkbox" 
                                           name="description_settings[include_technical]" 
                                           value="1" 
                                           <?php checked( isset( $description_settings['include_technical'] ) ? $description_settings['include_technical'] : '1', '1' ); ?>>
                                    <?php _e( 'Technical details (perforation, watermark, etc.)', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="description_settings[include_catalog_info]" 
                                           value="1" 
                                           <?php checked( isset( $description_settings['include_catalog_info'] ) ? $description_settings['include_catalog_info'] : '1', '1' ); ?>>
                                    <?php _e( 'Catalog information', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="description_settings[include_condition_details]" 
                                           value="1" 
                                           <?php checked( isset( $description_settings['include_condition_details'] ) ? $description_settings['include_condition_details'] : '1', '1' ); ?>>
                                    <?php _e( 'Detailed condition description', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="description_settings[include_shipping_info]" 
                                           value="1" 
                                           <?php checked( isset( $description_settings['include_shipping_info'] ) ? $description_settings['include_shipping_info'] : '1', '1' ); ?>>
                                    <?php _e( 'Shipping information', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="description_settings[include_payment_terms]" 
                                           value="1" 
                                           <?php checked( isset( $description_settings['include_payment_terms'] ) ? $description_settings['include_payment_terms'] : '1', '1' ); ?>>
                                    <?php _e( 'Payment terms', 'wc-delcampe-integration' ); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" 
                                           name="description_settings[include_guarantee]" 
                                           value="1" 
                                           <?php checked( isset( $description_settings['include_guarantee'] ) ? $description_settings['include_guarantee'] : '1', '1' ); ?>>
                                    <?php _e( 'Satisfaction guarantee', 'wc-delcampe-integration' ); ?>
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="custom_header">
                                <?php _e( 'Custom Header Text', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <textarea id="custom_header" 
                                      name="description_settings[custom_header]" 
                                      rows="3" 
                                      class="large-text"><?php echo esc_textarea( isset( $description_settings['custom_header'] ) ? $description_settings['custom_header'] : '' ); ?></textarea>
                            <p class="description">
                                <?php _e( 'Text to appear at the beginning of all descriptions (optional).', 'wc-delcampe-integration' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="custom_footer">
                                <?php _e( 'Custom Footer Text', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <textarea id="custom_footer" 
                                      name="description_settings[custom_footer]" 
                                      rows="3" 
                                      class="large-text"><?php echo esc_textarea( isset( $description_settings['custom_footer'] ) ? $description_settings['custom_footer'] : '' ); ?></textarea>
                            <p class="description">
                                <?php _e( 'Text to appear at the end of all descriptions (optional).', 'wc-delcampe-integration' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="description_language">
                                <?php _e( 'Description Language', 'wc-delcampe-integration' ); ?>
                            </label>
                        </th>
                        <td>
                            <select id="description_language" name="description_settings[language]">
                                <option value="en" <?php selected( isset( $description_settings['language'] ) ? $description_settings['language'] : 'en', 'en' ); ?>>
                                    <?php _e( 'English', 'wc-delcampe-integration' ); ?>
                                </option>
                                <option value="fr" <?php selected( isset( $description_settings['language'] ) ? $description_settings['language'] : 'en', 'fr' ); ?>>
                                    <?php _e( 'French', 'wc-delcampe-integration' ); ?>
                                </option>
                                <option value="de" <?php selected( isset( $description_settings['language'] ) ? $description_settings['language'] : 'en', 'de' ); ?>>
                                    <?php _e( 'German', 'wc-delcampe-integration' ); ?>
                                </option>
                                <option value="nl" <?php selected( isset( $description_settings['language'] ) ? $description_settings['language'] : 'en', 'nl' ); ?>>
                                    <?php _e( 'Dutch', 'wc-delcampe-integration' ); ?>
                                </option>
                            </select>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        <?php
        
        return ob_get_clean();
    }
}
