<?php
/**
 * Delcampe Listings List Table
 *
 * Displays the listings in a WordPress admin table format
 *
 * @package WooCommerce_Delcampe_Integration
 * @version 1.6.0.1
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Load WP_List_Table if not already loaded
if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

/**
 * Delcampe_Listings_List_Table class
 * 
 * Handles the display of Delcampe listings in a table format.
 * Extends WordPress core WP_List_Table class for consistency.
 * 
 * @since 1.6.0.0
 * @version 1.6.0.1
 */
class Delcampe_Listings_List_Table extends WP_List_Table {
    
    /**
     * Constructor
     * 
     * Sets up the list table with appropriate labels.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function __construct() {
        parent::__construct(array(
            'singular' => 'listing',
            'plural'   => 'listings',
            'ajax'     => false
        ));
    }
    
    /**
     * Get columns
     * 
     * Defines the columns to be displayed in the listings table.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @return array Column definitions
     */
    public function get_columns() {
        $columns = array(
            'cb'              => '<input type="checkbox" />',
            'img'             => __('Image', 'wc-delcampe-integration'),
            'listing_title'   => __('Title', 'wc-delcampe-integration'),
            'sku'             => __('SKU', 'wc-delcampe-integration'),
            'quantity'        => __('Quantity', 'wc-delcampe-integration'),
            'quantity_sold'   => __('Sold', 'wc-delcampe-integration'),
            'price'           => __('Price', 'wc-delcampe-integration'),
            'date_published'  => __('Published', 'wc-delcampe-integration'),
            'end_date'        => __('Ends', 'wc-delcampe-integration'),
            'profile'         => __('Profile', 'wc-delcampe-integration'),
            'status'          => __('Status', 'wc-delcampe-integration'),
        );
        
        return apply_filters('delcampe_listings_table_columns', $columns);
    }
    
    /**
     * Get sortable columns
     * 
     * Defines which columns can be sorted.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @return array Sortable column definitions
     */
    public function get_sortable_columns() {
        $sortable_columns = array(
            'listing_title'   => array('listing_title', false),
            'quantity'        => array('quantity', false),
            'quantity_sold'   => array('quantity_sold', false),
            'price'           => array('price', false),
            'date_published'  => array('date_published', false),
            'end_date'        => array('end_date', false),
            'status'          => array('status', false),
        );
        
        return $sortable_columns;
    }
    
    /**
     * Get bulk actions
     * 
     * Defines available bulk actions for listings.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @return array Bulk action definitions
     */
    public function get_bulk_actions() {
        $actions = array(
            'verify'       => __('Verify with Delcampe', 'wc-delcampe-integration'),
            'publish'      => __('Publish to Delcampe', 'wc-delcampe-integration'),
            'update'       => __('Update from Delcampe', 'wc-delcampe-integration'),
            'revise'       => __('Revise listings', 'wc-delcampe-integration'),
            'end'          => __('End listings', 'wc-delcampe-integration'),
            'relist'       => __('Relist ended items', 'wc-delcampe-integration'),
            'lock'         => __('Lock listings', 'wc-delcampe-integration'),
            'unlock'       => __('Unlock listings', 'wc-delcampe-integration'),
            'archive'      => __('Archive listings', 'wc-delcampe-integration'),
            'delete'       => __('Delete listings', 'wc-delcampe-integration'),
        );
        
        return $actions;
    }
    
    /**
     * Column callback: checkbox
     * 
     * Renders the checkbox for bulk actions.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="listing[]" value="%s" />',
            $item->id
        );
    }
    
    /**
     * Column callback: image
     * 
     * Renders the product image thumbnail.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_img($item) {
        $product_id = $item->product_id;
        if (!$product_id) {
            return '';
        }
        
        $product = wc_get_product($product_id);
        if (!$product) {
            return '';
        }
        
        $image = $product->get_image(array(50, 50));
        $preview_url = admin_url('admin.php?page=delcampe-listings&action=preview&listing=' . $item->id);
        
        return sprintf(
            '<a href="%s" class="thickbox">%s</a>',
            esc_url($preview_url),
            $image
        );
    }
    
    /**
     * Column callback: listing title
     * 
     * Renders the listing title with row actions.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_listing_title($item) {
        $actions = array();
        
        // Build actions based on status
        $page = isset($_REQUEST['page']) ? sanitize_text_field($_REQUEST['page']) : '';
        
        $actions['id'] = sprintf('ID: %d', $item->id);
        
        if ($item->delcampe_id) {
            $actions['delcampe_id'] = sprintf('Delcampe ID: %s', $item->delcampe_id);
        }
        
        $actions['preview'] = sprintf(
            '<a href="%s" class="thickbox">%s</a>',
            admin_url('admin.php?page=delcampe-listings&action=preview&listing=' . $item->id . '&TB_iframe=true&width=820&height=550'),
            __('Preview', 'wc-delcampe-integration')
        );
        
        $actions['edit'] = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=delcampe-listings&action=edit&listing=' . $item->id),
            __('Edit', 'wc-delcampe-integration')
        );
        
        // Status-specific actions
        switch ($item->status) {
            case 'prepared':
                $actions['verify'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=verify&listing=' . $item->id), 'verify_listing'),
                    __('Verify', 'wc-delcampe-integration')
                );
                break;
                
            case 'verified':
                $actions['publish'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=publish&listing=' . $item->id), 'publish_listing'),
                    __('Publish', 'wc-delcampe-integration')
                );
                break;
                
            case 'published':
                $actions['revise'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=revise&listing=' . $item->id), 'revise_listing'),
                    __('Revise', 'wc-delcampe-integration')
                );
                $actions['end'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=end&listing=' . $item->id), 'end_listing'),
                    __('End', 'wc-delcampe-integration')
                );
                
                if ($item->view_item_url) {
                    $actions['view'] = sprintf(
                        '<a href="%s" target="_blank">%s</a>',
                        esc_url($item->view_item_url),
                        __('View on Delcampe', 'wc-delcampe-integration')
                    );
                }
                break;
                
            case 'ended':
            case 'sold':
                $actions['relist'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=relist&listing=' . $item->id), 'relist_listing'),
                    __('Relist', 'wc-delcampe-integration')
                );
                $actions['archive'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=archive&listing=' . $item->id), 'archive_listing'),
                    __('Archive', 'wc-delcampe-integration')
                );
                break;
                
            case 'archived':
                $actions['delete'] = sprintf(
                    '<a href="%s">%s</a>',
                    wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=delete&listing=' . $item->id), 'delete_listing'),
                    __('Delete', 'wc-delcampe-integration')
                );
                break;
        }
        
        // Lock/unlock actions
        if ($item->locked) {
            $actions['unlock'] = sprintf(
                '<a href="%s">%s</a>',
                wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=unlock&listing=' . $item->id), 'unlock_listing'),
                __('Unlock', 'wc-delcampe-integration')
            );
        } else if ($item->status == 'published') {
            $actions['lock'] = sprintf(
                '<a href="%s">%s</a>',
                wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=lock&listing=' . $item->id), 'lock_listing'),
                __('Lock', 'wc-delcampe-integration')
            );
        }
        
        // Build title with edit link
        $title = $item->listing_title;
        if (strlen($title) > 80) {
            $title = substr($title, 0, 77) . '...';
        }
        
        $product_id = $item->parent_id ? $item->parent_id : $item->product_id;
        if ($product_id) {
            $title = sprintf(
                '<a href="%s">%s</a>',
                get_edit_post_link($product_id),
                esc_html($title)
            );
        }
        
        // Add lock indicator
        if ($item->locked) {
            $title .= ' <span class="dashicons dashicons-lock" title="' . __('This listing is locked', 'wc-delcampe-integration') . '"></span>';
        }
        
        // Check for errors
        if ($item->last_errors) {
            $errors = maybe_unserialize($item->last_errors);
            if (!empty($errors)) {
                $title .= ' <span class="dashicons dashicons-warning" style="color: #d54e21;" title="' . __('This listing has errors', 'wc-delcampe-integration') . '"></span>';
            }
        }
        
        return sprintf('%1$s %2$s', $title, $this->row_actions($actions));
    }
    
    /**
     * Column callback: SKU
     * 
     * Renders the product SKU.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_sku($item) {
        $sku = get_post_meta($item->product_id, '_sku', true);
        return $sku ? esc_html($sku) : '&mdash;';
    }
    
    /**
     * Column callback: quantity
     * 
     * Renders the listing quantity.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_quantity($item) {
        return intval($item->quantity);
    }
    
    /**
     * Column callback: quantity sold
     * 
     * Renders the quantity sold.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_quantity_sold($item) {
        return intval($item->quantity_sold);
    }
    
    /**
     * Column callback: price
     * 
     * Renders the listing price with currency.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_price($item) {
        $currency = $item->currency ? $item->currency : get_woocommerce_currency();
        return wc_price($item->price, array('currency' => $currency));
    }
    
    /**
     * Column callback: date published
     * 
     * Renders the publication date.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_date_published($item) {
        if (!$item->date_published || $item->date_published == '0000-00-00 00:00:00') {
            return '&mdash;';
        }
        
        return date_i18n(get_option('date_format'), strtotime($item->date_published));
    }
    
    /**
     * Column callback: end date
     * 
     * Renders the listing end date.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_end_date($item) {
        if (!$item->end_date || $item->end_date == '0000-00-00 00:00:00') {
            return '&mdash;';
        }
        
        return date_i18n(get_option('date_format'), strtotime($item->end_date));
    }
    
    /**
     * Column callback: profile
     * 
     * Renders the profile name with link.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_profile($item) {
        // Handle NULL or 0 profile_id - default to profile 2 (plate blocks)
        if (!$item->profile_id || $item->profile_id == 0) {
            // If we have a profile_name stored, use it
            if (!empty($item->profile_name) && $item->profile_name !== 'No profile') {
                return esc_html($item->profile_name);
            }
            
            // Otherwise default to plate blocks
            $default_profile_id = 2;
            $default_profile_name = 'plate blocks';
            
            // Update the database to fix this permanently
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'delcampe_listings',
                array('profile_id' => $default_profile_id),
                array('id' => $item->id),
                array('%d'),
                array('%d')
            );
            
            return esc_html($default_profile_name);
        }
        
        // Get profile name from profiles table
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_profiles';
        $profile_name = $wpdb->get_var($wpdb->prepare(
            "SELECT profile_name FROM $table_name WHERE profile_id = %d",
            $item->profile_id
        ));
        
        if ($profile_name) {
            $edit_url = admin_url('admin.php?page=delcampe-profiles&action=edit&profile=' . $item->profile_id);
            return sprintf(
                '<a href="%s">%s</a>',
                esc_url($edit_url),
                esc_html($profile_name)
            );
        }
        
        // If profile not found but we have a stored name, use it
        if (!empty($item->profile_name)) {
            return esc_html($item->profile_name);
        }
        
        return '<span style="color: #d54e21;">' . __('Profile missing', 'wc-delcampe-integration') . '</span>';
    }
    
    /**
     * Column callback: status
     * 
     * Renders the listing status with color coding.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @return string HTML output
     */
    public function column_status($item) {
        $status_labels = array(
            'prepared'  => __('Prepared', 'wc-delcampe-integration'),
            'verified'  => __('Verified', 'wc-delcampe-integration'),
            'published' => __('Published', 'wc-delcampe-integration'),
            'changed'   => __('Changed', 'wc-delcampe-integration'),
            'ended'     => __('Ended', 'wc-delcampe-integration'),
            'sold'      => __('Sold', 'wc-delcampe-integration'),
            'archived'  => __('Archived', 'wc-delcampe-integration'),
            'error'     => __('Error', 'wc-delcampe-integration'),
        );
        
        $status_colors = array(
            'prepared'  => '#f39c12',
            'verified'  => '#3498db',
            'published' => '#27ae60',
            'changed'   => '#9b59b6',
            'ended'     => '#7f8c8d',
            'sold'      => '#2c3e50',
            'archived'  => '#95a5a6',
            'error'     => '#e74c3c',
        );
        
        $label = isset($status_labels[$item->status]) ? $status_labels[$item->status] : $item->status;
        $color = isset($status_colors[$item->status]) ? $status_colors[$item->status] : '#333';
        
        return sprintf(
            '<mark style="background-color: %s; color: #fff; padding: 3px 8px; border-radius: 3px;">%s</mark>',
            esc_attr($color),
            esc_html($label)
        );
    }
    
    /**
     * Default column callback
     * 
     * Handles any columns not specifically defined.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param object $item Listing object
     * @param string $column_name Column name
     * @return string HTML output
     */
    public function column_default($item, $column_name) {
        return isset($item->$column_name) ? esc_html($item->$column_name) : '';
    }
    
    /**
     * Get views (status filters)
     * 
     * Creates filter links for different listing statuses.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @return array View links
     */
    public function get_views() {
        $views = array();
        $current = isset($_REQUEST['listing_status']) ? $_REQUEST['listing_status'] : 'all';
        
        // Get status summary
        $summary = Delcampe_Listings_Model::get_status_summary();
        
        // All link
        $class = ($current == 'all') ? ' class="current"' : '';
        $url = remove_query_arg('listing_status');
        $views['all'] = sprintf(
            '<a href="%s"%s>%s <span class="count">(%d)</span></a>',
            esc_url($url),
            $class,
            __('All', 'wc-delcampe-integration'),
            $summary->total_items
        );
        
        // Status links
        $statuses = array(
            'prepared'  => array('label' => __('Prepared', 'wc-delcampe-integration'), 'count' => $summary->prepared),
            'verified'  => array('label' => __('Verified', 'wc-delcampe-integration'), 'count' => $summary->verified),
            'published' => array('label' => __('Published', 'wc-delcampe-integration'), 'count' => $summary->published),
            'changed'   => array('label' => __('Changed', 'wc-delcampe-integration'), 'count' => $summary->changed),
            'ended'     => array('label' => __('Ended', 'wc-delcampe-integration'), 'count' => $summary->ended),
            'sold'      => array('label' => __('Sold', 'wc-delcampe-integration'), 'count' => $summary->sold),
            'archived'  => array('label' => __('Archived', 'wc-delcampe-integration'), 'count' => $summary->archived),
        );
        
        foreach ($statuses as $status => $data) {
            if ($data['count'] > 0) {
                $class = ($current == $status) ? ' class="current"' : '';
                $url = add_query_arg('listing_status', $status);
                $views[$status] = sprintf(
                    '<a href="%s"%s>%s <span class="count">(%d)</span></a>',
                    esc_url($url),
                    $class,
                    $data['label'],
                    $data['count']
                );
            }
        }
        
        // With errors
        if ($summary->with_errors > 0) {
            $class = ($current == 'error') ? ' class="current"' : '';
            $url = add_query_arg('listing_status', 'error');
            $views['error'] = sprintf(
                '<a href="%s"%s>%s <span class="count">(%d)</span></a>',
                esc_url($url),
                $class,
                __('With Errors', 'wc-delcampe-integration'),
                $summary->with_errors
            );
        }
        
        // Locked
        if ($summary->locked > 0) {
            $class = ($current == 'locked') ? ' class="current"' : '';
            $url = add_query_arg('listing_status', 'locked');
            $views['locked'] = sprintf(
                '<a href="%s"%s>%s <span class="count">(%d)</span></a>',
                esc_url($url),
                $class,
                __('Locked', 'wc-delcampe-integration'),
                $summary->locked
            );
        }
        
        return $views;
    }
    
    /**
     * Process bulk actions
     * 
     * Handles bulk action processing for selected listings.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function process_bulk_action() {
        if (!$this->current_action()) {
            return;
        }
        
        // Security check
        if (!isset($_REQUEST['_wpnonce']) || !wp_verify_nonce($_REQUEST['_wpnonce'], 'bulk-' . $this->_args['plural'])) {
            wp_die(__('Security check failed', 'wc-delcampe-integration'));
        }
        
        $listing_ids = isset($_REQUEST['listing']) ? array_map('intval', $_REQUEST['listing']) : array();
        
        if (empty($listing_ids)) {
            return;
        }
        
        $action = $this->current_action();
        $processed = 0;
        
        foreach ($listing_ids as $listing_id) {
            switch ($action) {
                case 'delete':
                    if (Delcampe_Listings_Model::delete_listing($listing_id)) {
                        $processed++;
                    }
                    break;
                    
                case 'archive':
                    if (Delcampe_Listings_Model::update_status($listing_id, 'archived')) {
                        $processed++;
                    }
                    break;
                    
                case 'lock':
                    if (Delcampe_Listings_Model::set_locked($listing_id, true)) {
                        $processed++;
                    }
                    break;
                    
                case 'unlock':
                    if (Delcampe_Listings_Model::set_locked($listing_id, false)) {
                        $processed++;
                    }
                    break;
                    
                case 'end':
                    // End listing on Delcampe
                    $listing = Delcampe_Listings_Model::get_listing($listing_id);
                    if ($listing && !empty($listing->delcampe_id)) {
                        // Load the API class
                        require_once plugin_dir_path(dirname(dirname(__FILE__))) . 'includes/class-delcampe-listing-api.php';
                        $api = Delcampe_Listing_API::get_instance();
                        
                        // Call Delcampe API to close the listing
                        $result = $api->close_listing((string)$listing->delcampe_id, 'NotAvailableItem');
                        
                        if (!is_wp_error($result) && !empty($result['success'])) {
                            // Update local status
                            Delcampe_Listings_Model::update_listing($listing_id, array(
                                'status' => 'ended',
                                'date_finished' => current_time('mysql'),
                            ));
                            
                            // Clean up Delcampe IDs from product meta
                            if ($listing->product_id) {
                                delete_post_meta($listing->product_id, '_delcampe_item_id');
                                delete_post_meta($listing->product_id, '_delcampe_listing_id');
                                delcampe_log(sprintf('[Bulk End] Cleaned up IDs for product #%d', $listing->product_id));
                            }
                            
                            $processed++;
                        }
                    } elseif ($listing && empty($listing->delcampe_id)) {
                        // Fallback: try to get delcampe_id from product meta
                        $delcampe_id = get_post_meta((int)$listing->product_id, '_delcampe_listing_id', true);
                        if ($delcampe_id) {
                            require_once plugin_dir_path(dirname(dirname(__FILE__))) . 'includes/class-delcampe-listing-api.php';
                            $api = Delcampe_Listing_API::get_instance();
                            $result = $api->close_listing((string)$delcampe_id, 'NotAvailableItem');
                            
                            if (!is_wp_error($result) && !empty($result['success'])) {
                                Delcampe_Listings_Model::update_listing($listing_id, array(
                                    'status' => 'ended',
                                    'date_finished' => current_time('mysql'),
                                ));
                                
                                // Clean up Delcampe IDs from product meta
                                if ($listing->product_id) {
                                    delete_post_meta($listing->product_id, '_delcampe_item_id');
                                    delete_post_meta($listing->product_id, '_delcampe_listing_id');
                                    delcampe_log(sprintf('[Bulk End] Cleaned up IDs for product #%d', $listing->product_id));
                                }
                                
                                $processed++;
                            }
                        }
                    }
                    break;
                    
                // Add more bulk action handlers as needed
            }
        }
        
        if ($processed > 0) {
            $message = sprintf(
                _n('%d listing processed.', '%d listings processed.', $processed, 'wc-delcampe-integration'),
                $processed
            );
            
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($message) . '</p></div>';
        }
    }
    
    /**
     * Prepare items for display
     * 
     * Prepares the data for display in the table.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function prepare_items() {
        // Process bulk actions
        $this->process_bulk_action();
        
        // Set column headers
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        
        $this->_column_headers = array($columns, $hidden, $sortable);
        
        // Get data
        $per_page = $this->get_items_per_page('delcampe_listings_per_page', 50); // Default 50 instead of 20
        $current_page = $this->get_pagenum();
        
        // Build query args
        $args = array(
            'limit' => $per_page,
            'offset' => ($current_page - 1) * $per_page,
        );
        
        // Add status filter
        if (isset($_REQUEST['listing_status']) && $_REQUEST['listing_status'] != 'all') {
            $status = sanitize_text_field($_REQUEST['listing_status']);
            
            if ($status == 'locked') {
                // Special handling for locked filter
                $args['locked'] = true;
            } elseif ($status == 'error') {
                // Special handling for error filter
                $args['status'] = 'error';
            } else {
                $args['status'] = $status;
            }
        }
        
        // Add search
        if (!empty($_REQUEST['s'])) {
            $args['search'] = sanitize_text_field($_REQUEST['s']);
        }
        
        // Add sorting
        if (!empty($_REQUEST['orderby'])) {
            $args['orderby'] = sanitize_text_field($_REQUEST['orderby']);
            $args['order'] = !empty($_REQUEST['order']) ? sanitize_text_field($_REQUEST['order']) : 'ASC';
        }
        
        // Get items
        $this->items = Delcampe_Listings_Model::get_listings($args);
        
        // Get total count
        $total_items = Delcampe_Listings_Model::get_listings_count($args);
        
        // Set pagination
        $this->set_pagination_args(array(
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ));
    }
}
