<?php
/**
 * Delcampe Orders List Table
 * 
 * Extends WP_List_Table to display Delcampe orders
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin
 * @since 1.4.0.0
 * @version 1.4.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

// Include WP_List_Table if not already loaded
if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

/**
 * Class Delcampe_Orders_List_Table
 * 
 * Displays Delcampe orders in a WordPress admin table
 * 
 * @since 1.4.0.0
 */
class Delcampe_Orders_List_Table extends WP_List_Table {
    
    /**
     * Constructor
     * 
     * @since 1.4.0.0
     */
    public function __construct() {
        parent::__construct(array(
            'singular' => 'delcampe_order',
            'plural' => 'delcampe_orders',
            'ajax' => false
        ));
    }
    
    /**
     * Get columns
     * 
     * @return array Column definitions
     * 
     * @since 1.4.0.0
     */
    public function get_columns() {
        return array(
            'cb' => '<input type="checkbox" />',
            'delcampe_order_id' => __('Delcampe ID', 'wc-delcampe-integration'),
            'wc_order' => __('WC Order', 'wc-delcampe-integration'),
            'buyer' => __('Buyer', 'wc-delcampe-integration'),
            'product' => __('Product', 'wc-delcampe-integration'),
            'sale_price' => __('Price', 'wc-delcampe-integration'),
            'sale_date' => __('Sale Date', 'wc-delcampe-integration'),
            'payment_status' => __('Payment', 'wc-delcampe-integration'),
            'shipping_status' => __('Shipping', 'wc-delcampe-integration'),
            'sync_status' => __('Sync Status', 'wc-delcampe-integration'),
            'actions' => __('Actions', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Get sortable columns
     * 
     * @return array Sortable column definitions
     * 
     * @since 1.4.0.0
     */
    public function get_sortable_columns() {
        return array(
            'delcampe_order_id' => array('delcampe_order_id', false),
            'wc_order' => array('wc_order_id', false),
            'buyer' => array('buyer_nickname', false),
            'sale_price' => array('sale_price', false),
            'sale_date' => array('sale_date', true), // Default descending
            'payment_status' => array('payment_status', false),
            'shipping_status' => array('shipping_status', false),
            'sync_status' => array('sync_status', false)
        );
    }
    
    /**
     * Get bulk actions
     * 
     * @return array Bulk action definitions
     * 
     * @since 1.4.0.0
     */
    public function get_bulk_actions() {
        return array(
            'sync' => __('Sync Orders', 'wc-delcampe-integration'),
            'create_wc_orders' => __('Create WC Orders', 'wc-delcampe-integration'),
            'update_status' => __('Update Status', 'wc-delcampe-integration'),
            'delete' => __('Delete Orders', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Checkbox column
     * 
     * @param object $item Row data
     * @return string Checkbox HTML
     * 
     * @since 1.10.13.16
     */
    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="order[]" value="%d" />',
            $item->id
        );
    }
    
    /**
     * Column default
     * 
     * @param object $item Row data
     * @param string $column_name Column name
     * @return string Column content
     * 
     * @since 1.4.0.0
     */
    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'delcampe_order_id':
                return esc_html($item->delcampe_order_id);
                
            case 'wc_order':
                if ($item->wc_order_id) {
                    $order = wc_get_order($item->wc_order_id);
                    if ($order) {
                        return sprintf(
                            '<a href="%s">#%d</a>',
                            $order->get_edit_order_url(),
                            $item->wc_order_id
                        );
                    }
                }
                return '<span class="dashicons dashicons-minus"></span>';
                
            case 'buyer':
                return sprintf(
                    '%s<br><small>%s</small>',
                    esc_html($item->buyer_nickname),
                    esc_html($item->buyer_email)
                );
                
            case 'product':
                if ($item->wc_product_id) {
                    $product = wc_get_product($item->wc_product_id);
                    if ($product) {
                        return sprintf(
                            '<a href="%s">%s</a>',
                            get_edit_post_link($item->wc_product_id),
                            esc_html($product->get_name())
                        );
                    }
                }
                return '<em>' . __('Not linked', 'wc-delcampe-integration') . '</em>';
                
            case 'sale_price':
                return wc_price($item->sale_price) . ' ' . esc_html($item->currency);
                
            case 'sale_date':
                return date_i18n(
                    get_option('date_format') . ' ' . get_option('time_format'),
                    strtotime($item->sale_date)
                );
                
            case 'payment_status':
                return $this->format_status($item->payment_status, 'payment');
                
            case 'shipping_status':
                return $this->format_status($item->shipping_status, 'shipping');
                
            case 'sync_status':
                return $this->format_status($item->sync_status, 'sync');
                
            case 'actions':
                $actions = array();
                
                // Sync action
                $actions[] = sprintf(
                    '<a href="#" class="button button-small sync-order" data-order-id="%d">%s</a>',
                    $item->id,
                    __('Sync', 'wc-delcampe-integration')
                );
                
                // View details
                $actions[] = sprintf(
                    '<a href="#" class="button button-small view-details" data-order-id="%d">%s</a>',
                    $item->id,
                    __('Details', 'wc-delcampe-integration')
                );
                
                return implode(' ', $actions);
                
            default:
                return print_r($item, true);
        }
    }
    
    
    /**
     * Format status display
     * 
     * @param string $status Status value
     * @param string $type Status type
     * @return string Formatted status HTML
     * 
     * @since 1.4.0.0
     */
    private function format_status($status, $type) {
        $status_classes = array(
            'pending' => 'status-pending',
            'processing' => 'status-processing',
            'synced' => 'status-success',
            'completed' => 'status-success',
            'paid' => 'status-success',
            'shipped' => 'status-info',
            'failed' => 'status-error',
            'refunded' => 'status-warning'
        );
        
        $class = isset($status_classes[$status]) ? $status_classes[$status] : 'status-default';
        
        return sprintf(
            '<span class="status-badge %s">%s</span>',
            esc_attr($class),
            esc_html(ucfirst($status))
        );
    }
    
    /**
     * Process bulk actions
     * 
     * @since 1.10.13.16
     */
    public function process_bulk_action() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_orders';
        
        // Security check
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('You do not have permission to perform this action.', 'wc-delcampe-integration'));
        }
        
        // Check for bulk delete action
        if ('delete' === $this->current_action()) {
            // Verify nonce
            $nonce = isset($_REQUEST['_wpnonce']) ? $_REQUEST['_wpnonce'] : '';
            if (!wp_verify_nonce($nonce, 'bulk-' . $this->_args['plural'])) {
                wp_die(__('Security check failed', 'wc-delcampe-integration'));
            }
            
            // Get selected order IDs
            $order_ids = isset($_REQUEST['order']) ? array_map('intval', $_REQUEST['order']) : array();
            
            if (!empty($order_ids)) {
                $deleted_count = 0;
                
                foreach ($order_ids as $order_id) {
                    // Get the order details first
                    $order_data = $wpdb->get_row($wpdb->prepare(
                        "SELECT * FROM $table_name WHERE id = %d",
                        $order_id
                    ));
                    
                    if ($order_data) {
                        // Delete the WooCommerce order if it exists
                        if ($order_data->wc_order_id) {
                            $wc_order = wc_get_order($order_data->wc_order_id);
                            if ($wc_order) {
                                // Optional: You can choose to keep or delete the WC order
                                // Comment out the next line if you want to keep WC orders
                                // $wc_order->delete(true); // true = force delete (bypass trash)
                                
                                // Just add a note that Delcampe sync was removed
                                $wc_order->add_order_note(__('Delcampe sync record deleted', 'wc-delcampe-integration'));
                            }
                        }
                        
                        // Delete from Delcampe orders table
                        $result = $wpdb->delete(
                            $table_name,
                            array('id' => $order_id),
                            array('%d')
                        );
                        
                        if ($result !== false) {
                            $deleted_count++;
                            
                            // Log the deletion
                            if (function_exists('delcampe_log')) {
                                delcampe_log(sprintf(
                                    '[Orders] Deleted order record #%d (Delcampe ID: %s, WC Order: %s)',
                                    $order_id,
                                    $order_data->delcampe_order_id,
                                    $order_data->wc_order_id ?: 'none'
                                ));
                            }
                        }
                    }
                }
                
                // Add admin notice
                if ($deleted_count > 0) {
                    add_settings_error(
                        'delcampe_orders',
                        'orders_deleted',
                        sprintf(
                            _n(
                                '%d order deleted successfully.',
                                '%d orders deleted successfully.',
                                $deleted_count,
                                'wc-delcampe-integration'
                            ),
                            $deleted_count
                        ),
                        'success'
                    );
                }
            }
        }
        
        // Handle other bulk actions
        if ('sync' === $this->current_action()) {
            // Existing sync logic if any
        }
        
        if ('create_wc_orders' === $this->current_action()) {
            // Logic to create WC orders for selected items
        }
        
        if ('update_status' === $this->current_action()) {
            // Logic to update status
        }
    }
    
    /**
     * Prepare items
     * 
     * @since 1.4.0.0
     */
    public function prepare_items() {
        global $wpdb;
        
        // Process bulk actions first
        $this->process_bulk_action();
        
        // Set column headers
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        
        $this->_column_headers = array($columns, $hidden, $sortable);
        
        // Get per page
        $per_page = $this->get_items_per_page('delcampe_orders_per_page', 20);
        $current_page = $this->get_pagenum();
        
        // Build query
        $table_name = $wpdb->prefix . 'delcampe_orders';
        
        // Handle search
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';
        $where = '';
        
        if ($search) {
            $where = $wpdb->prepare(
                " WHERE delcampe_order_id LIKE %s 
                OR buyer_nickname LIKE %s 
                OR buyer_email LIKE %s",
                '%' . $wpdb->esc_like($search) . '%',
                '%' . $wpdb->esc_like($search) . '%',
                '%' . $wpdb->esc_like($search) . '%'
            );
        }
        
        // Handle filters
        if (!empty($_REQUEST['payment_status'])) {
            $payment_filter = sanitize_text_field($_REQUEST['payment_status']);
            $where .= $where ? ' AND ' : ' WHERE ';
            $where .= $wpdb->prepare('payment_status = %s', $payment_filter);
        }
        
        if (!empty($_REQUEST['shipping_status'])) {
            $shipping_filter = sanitize_text_field($_REQUEST['shipping_status']);
            $where .= $where ? ' AND ' : ' WHERE ';
            $where .= $wpdb->prepare('shipping_status = %s', $shipping_filter);
        }
        
        if (!empty($_REQUEST['sync_status'])) {
            $sync_filter = sanitize_text_field($_REQUEST['sync_status']);
            $where .= $where ? ' AND ' : ' WHERE ';
            $where .= $wpdb->prepare('sync_status = %s', $sync_filter);
        }
        
        // Get total items
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name $where");
        
        // Handle ordering
        $orderby = !empty($_REQUEST['orderby']) ? sanitize_sql_orderby($_REQUEST['orderby']) : 'sale_date';
        $order = !empty($_REQUEST['order']) ? sanitize_text_field($_REQUEST['order']) : 'DESC';
        
        // Get items
        $offset = ($current_page - 1) * $per_page;
        $items = $wpdb->get_results(
            "SELECT * FROM $table_name 
            $where 
            ORDER BY $orderby $order 
            LIMIT $offset, $per_page"
        );
        
        // Set items
        $this->items = $items;
        
        // Set pagination
        $this->set_pagination_args(array(
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ));
    }
    
    /**
     * Extra table navigation
     * 
     * @param string $which Top or bottom
     * 
     * @since 1.4.0.0
     */
    protected function extra_tablenav($which) {
        if ($which !== 'top') {
            return;
        }
        
        ?>
        <div class="alignleft actions">
            <!-- Payment Status Filter -->
            <select name="payment_status" id="filter-payment-status">
                <option value=""><?php _e('All Payment Status', 'wc-delcampe-integration'); ?></option>
                <option value="pending" <?php selected(isset($_REQUEST['payment_status']) && $_REQUEST['payment_status'] === 'pending'); ?>>
                    <?php _e('Pending', 'wc-delcampe-integration'); ?>
                </option>
                <option value="paid" <?php selected(isset($_REQUEST['payment_status']) && $_REQUEST['payment_status'] === 'paid'); ?>>
                    <?php _e('Paid', 'wc-delcampe-integration'); ?>
                </option>
                <option value="refunded" <?php selected(isset($_REQUEST['payment_status']) && $_REQUEST['payment_status'] === 'refunded'); ?>>
                    <?php _e('Refunded', 'wc-delcampe-integration'); ?>
                </option>
            </select>
            
            <!-- Shipping Status Filter -->
            <select name="shipping_status" id="filter-shipping-status">
                <option value=""><?php _e('All Shipping Status', 'wc-delcampe-integration'); ?></option>
                <option value="pending" <?php selected(isset($_REQUEST['shipping_status']) && $_REQUEST['shipping_status'] === 'pending'); ?>>
                    <?php _e('Pending', 'wc-delcampe-integration'); ?>
                </option>
                <option value="processing" <?php selected(isset($_REQUEST['shipping_status']) && $_REQUEST['shipping_status'] === 'processing'); ?>>
                    <?php _e('Processing', 'wc-delcampe-integration'); ?>
                </option>
                <option value="shipped" <?php selected(isset($_REQUEST['shipping_status']) && $_REQUEST['shipping_status'] === 'shipped'); ?>>
                    <?php _e('Shipped', 'wc-delcampe-integration'); ?>
                </option>
                <option value="completed" <?php selected(isset($_REQUEST['shipping_status']) && $_REQUEST['shipping_status'] === 'completed'); ?>>
                    <?php _e('Completed', 'wc-delcampe-integration'); ?>
                </option>
            </select>
            
            <!-- Sync Status Filter -->
            <select name="sync_status" id="filter-sync-status">
                <option value=""><?php _e('All Sync Status', 'wc-delcampe-integration'); ?></option>
                <option value="pending" <?php selected(isset($_REQUEST['sync_status']) && $_REQUEST['sync_status'] === 'pending'); ?>>
                    <?php _e('Pending', 'wc-delcampe-integration'); ?>
                </option>
                <option value="synced" <?php selected(isset($_REQUEST['sync_status']) && $_REQUEST['sync_status'] === 'synced'); ?>>
                    <?php _e('Synced', 'wc-delcampe-integration'); ?>
                </option>
                <option value="failed" <?php selected(isset($_REQUEST['sync_status']) && $_REQUEST['sync_status'] === 'failed'); ?>>
                    <?php _e('Failed', 'wc-delcampe-integration'); ?>
                </option>
            </select>
            
            <?php submit_button(__('Filter', 'wc-delcampe-integration'), 'button', 'filter_action', false); ?>
        </div>
        <?php
    }
    
    /**
     * No items message
     * 
     * @since 1.4.0.0
     */
    public function no_items() {
        _e('No Delcampe orders found. Click "Import Orders" to fetch orders from Delcampe.', 'wc-delcampe-integration');
    }
}
