<?php
/**
 * Delcampe Profiles List Table
 * Version: 1.1.3.0
 *
 * Extends WP_List_Table to display profiles in the admin area
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Tables
 * @since 1.1.3.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Load WP_List_Table if not loaded
if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class Delcampe_Profiles_List_Table
 * 
 * Displays Delcampe profiles in a standard WordPress admin table
 * 
 * @version 1.1.3.0
 */
class Delcampe_Profiles_List_Table extends WP_List_Table {

    /**
     * Profiles model instance
     * @var Delcampe_Profiles_Model
     */
    private $profiles_model;

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct( array(
            'singular' => 'profile',
            'plural' => 'profiles',
            'ajax' => false,
        ) );

        $this->profiles_model = Delcampe_Profiles_Model::get_instance();
    }

    /**
     * Define table columns
     * 
     * @return array Column definitions
     */
    public function get_columns() {
        return array(
            'cb' => '<input type="checkbox" />',
            'profile_name' => __( 'Profile Name', 'wc-delcampe-integration' ),
            'woo_category' => __( 'WooCommerce Category', 'wc-delcampe-integration' ),
            'delcampe_category' => __( 'Delcampe Category', 'wc-delcampe-integration' ),
            'products_count' => __( 'Products', 'wc-delcampe-integration' ),
            'is_default' => __( 'Default', 'wc-delcampe-integration' ),
            'date_modified' => __( 'Last Modified', 'wc-delcampe-integration' ),
        );
    }

    /**
     * Define sortable columns
     * 
     * @return array Sortable column definitions
     */
    public function get_sortable_columns() {
        return array(
            'profile_name' => array( 'profile_name', false ),
            'woo_category' => array( 'woo_category_name', false ),
            'delcampe_category' => array( 'delcampe_category_name', false ),
            'is_default' => array( 'is_default', false ),
            'date_modified' => array( 'date_modified', false ),
        );
    }

    /**
     * Prepare items for display
     * 
     * @return void
     */
    public function prepare_items() {
        // Set column headers
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        
        $this->_column_headers = array( $columns, $hidden, $sortable );

        // Handle bulk actions
        $this->process_bulk_action();

        // Get data
        $per_page = $this->get_items_per_page( 'profiles_per_page', 20 );
        $current_page = $this->get_pagenum();
        
        // Get profiles
        $this->items = $this->profiles_model->get_page_items( $current_page, $per_page );
        $total_items = $this->profiles_model->total_items;

        // Set pagination
        $this->set_pagination_args( array(
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil( $total_items / $per_page ),
        ) );
    }

    /**
     * Default column output
     * 
     * @param array  $item        Current item
     * @param string $column_name Column name
     * @return string Column content
     */
    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'woo_category':
                return ! empty( $item['woo_category_name'] ) ? 
                    esc_html( $item['woo_category_name'] ) : 
                    '<em>' . __( 'Not set', 'wc-delcampe-integration' ) . '</em>';
                    
            case 'delcampe_category':
                if ( ! empty( $item['delcampe_category_name'] ) ) {
                    $output = esc_html( $item['delcampe_category_name'] );
                    if ( ! empty( $item['delcampe_category_path'] ) ) {
                        $output .= '<br><small style="color:#666;">' . esc_html( $item['delcampe_category_path'] ) . '</small>';
                    }
                    return $output;
                }
                return '<em>' . __( 'Not set', 'wc-delcampe-integration' ) . '</em>';
                
            case 'products_count':
                $count = count( $this->profiles_model->get_products_using_profile( $item['profile_id'] ) );
                if ( $count > 0 ) {
                    $url = admin_url( 'edit.php?post_type=product&delcampe_profile_id=' . $item['profile_id'] );
                    return '<a href="' . esc_url( $url ) . '">' . $count . '</a>';
                }
                return '0';
                
            case 'is_default':
                return $item['is_default'] ? 
                    '<span class="dashicons dashicons-star-filled" style="color:#0073aa;"></span>' : 
                    '';
                    
            case 'date_modified':
                return date_i18n( 
                    get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), 
                    strtotime( $item['date_modified'] ) 
                );
                
            default:
                return isset( $item[ $column_name ] ) ? $item[ $column_name ] : '';
        }
    }

    /**
     * Checkbox column
     * 
     * @param array $item Current item
     * @return string Checkbox HTML
     */
    public function column_cb( $item ) {
        return sprintf(
            '<input type="checkbox" name="profile_ids[]" value="%d" />',
            $item['profile_id']
        );
    }

    /**
     * Profile name column with actions
     * 
     * @param array $item Current item
     * @return string Column content with actions
     */
    public function column_profile_name( $item ) {
        // Build actions
        $actions = array(
            'edit' => sprintf(
                '<a href="%s">%s</a>',
                admin_url( 'admin.php?page=delcampe-profiles&action=edit&profile_id=' . $item['profile_id'] ),
                __( 'Edit', 'wc-delcampe-integration' )
            ),
            'duplicate' => sprintf(
                '<a href="%s">%s</a>',
                admin_url( 'admin.php?page=delcampe-profiles&action=duplicate&profile_id=' . $item['profile_id'] ),
                __( 'Duplicate', 'wc-delcampe-integration' )
            ),
            'delete' => sprintf(
                '<a href="%s" onclick="return confirm(\'%s\');">%s</a>',
                wp_nonce_url(
                    admin_url( 'admin.php?page=delcampe-profiles&action=delete&profile_id=' . $item['profile_id'] ),
                    'delete-profile_' . $item['profile_id']
                ),
                esc_attr__( 'Are you sure you want to delete this profile?', 'wc-delcampe-integration' ),
                __( 'Delete', 'wc-delcampe-integration' )
            ),
        );

        // Build title
        $title = sprintf(
            '<strong><a href="%s">%s</a></strong>',
            admin_url( 'admin.php?page=delcampe-profiles&action=edit&profile_id=' . $item['profile_id'] ),
            esc_html( $item['profile_name'] )
        );

        // Add description if exists
        if ( ! empty( $item['profile_description'] ) ) {
            $title .= '<br><small>' . esc_html( $item['profile_description'] ) . '</small>';
        }

        return $title . $this->row_actions( $actions );
    }

    /**
     * Define bulk actions
     * 
     * @return array Bulk action definitions
     */
    public function get_bulk_actions() {
        return array(
            'delete' => __( 'Delete', 'wc-delcampe-integration' ),
        );
    }

    /**
     * Process bulk actions
     * 
     * @return void
     */
    public function process_bulk_action() {
        // Security check
        if ( 'delete' === $this->current_action() ) {
            // Handled in main admin class
        }
    }

    /**
     * Display when no items found
     * 
     * @return void
     */
    public function no_items() {
        _e( 'No profiles found.', 'wc-delcampe-integration' );
        echo ' <a href="' . admin_url( 'admin.php?page=delcampe-profiles&action=new' ) . '">';
        _e( 'Create your first profile', 'wc-delcampe-integration' );
        echo '</a>.';
    }

    /**
     * Extra controls above/below the table
     * 
     * @param string $which Position (top or bottom)
     * @return void
     */
    public function extra_tablenav( $which ) {
        if ( 'top' === $which ) {
            // Could add filters here if needed
        }
    }
}
