<?php
/**
 * Plugin Name: Delcampe Sync
 * Plugin URI:  https://www.sos-tech.ca/plugins/delcampe-sync
 * Description: Seamlessly sync your WooCommerce products with Delcampe marketplace. Automated listings, order management, and inventory synchronization.
 * Version:     1.10.37.0
 * Author:      SOS Services
 * Author URI:  https://www.sos-tech.ca
 * License:     Proprietary
 * License URI: https://www.sos-tech.ca/delcampe-sync-license/
 * Text Domain: wc-delcampe-integration
 * Domain Path: /languages
 * 
 * WC requires at least: 5.0
 * WC tested up to: 8.9
 * 
 * @package WooCommerce_Delcampe_Integration
 * @author SOS Services
 * @since 1.0.0
 * 
 * This plugin facilitates seamless communication between WooCommerce (master) and Delcampe platforms.
 * All API communication with Delcampe uses XML format for data exchange.
 * 
 * Changelog:
 * 1.10.8.0 - 2025-08-06 (Trial System & License Updates)
 *           - Implemented 14-day free trial system with 5 listing limit
 *           - Added automatic status checking for processing listings every 10 minutes
 *           - Aligned license tiers with pricing model
 *           - Changed from GPL to Proprietary licensing
 *           - Restricted Import from Delcampe to Professional/Enterprise tiers
 *           - Limited Enterprise to 10 site licenses (from unlimited)
 *           - Added trial status display in admin notices and license page
 * 1.10.5.17 - 2025-08-05 (Webhook Manager Implementation)
 *           - Added programmatic webhook registration via Delcampe API
 *           - Created webhook manager for automated notification setup
 *           - Added admin interface for webhook configuration
 *           - Supports all Delcampe notification types (item sold, updated, etc.)
 * 1.10.5.16 - 2025-08-05 (Real-time Order Processing)
 *           - Implemented webhook handler for immediate order creation
 *           - Fixed listing status logic to only mark 'sold' when qty <= 0
 *           - Made import_single_order() public for webhook access
 *           - Added create_order_from_webhook() method
 * 1.10.5.15 - 2025-08-05 (Order Processing Verification)
 *           - Verified order processing and inventory reduction functionality
 *           - Found and documented webhook handler stub for real-time processing
 * 1.10.5.14 - 2025-08-05 (Shipping Logs Fix)
 *           - Fixed shipping model logs to use plugin logging system
 *           - Changed error_log() to delcampe_log() in shipping models class
 * 1.10.5.13 - 2025-08-05 (Translation Warning Final Fix)
 *           - Fixed remaining translation loading warnings in cron schedules
 *           - Fixed translation calls in plugin header data
 *           - Completely eliminated translation loading warnings
 * 1.10.5.12 - 2025-08-05 (Additional Translation Fixes)
 *           - Fixed Sync Handler translation loading issues
 *           - Fixed Updater class translation loading issues
 *           - Deferred more hooks to 'init' action
 * 1.10.5.11 - 2025-08-04 (Translation Loading Fix)
 *           - Fixed wrong text domain in stamp profile fields
 *           - Fixed wrong text domain in import page render
 *           - Added conditional translation for plugin action links
 *           - Removed translation calls from activation hook
 * 1.10.5.10 - 2025-08-04 (Admin Notice Deferral)
 *           - Deferred admin notices to init hook for WordPress 6.7 compatibility
 * 1.10.5.5 - 2025-08-04 (Menu Reorganization)
 *          - Moved Cleanup functionality to Inventory → Cleanup tab
 *          - Moved License page to Settings → License tab
 *          - Improved menu organization for better UX
 * 1.10.5.4 - 2025-08-04 (Update System Fix)
 *          - Fixed "A valid URL was not provided" error
 *          - Update server now always provides download URL
 *          - License validation happens during download, not URL generation
 *          - Better separation of concerns: server provides files, client validates license
 * 1.10.5.3 - 2025-08-03 (DELETE Method Fix)
 *          - Corrected to use DELETE method with query parameters
 *          - Removed XML body, now uses URL parameters: ?token=X&reason=Y&priority=1
 *          - Matches actual Delcampe ItemDelete API documentation
 *          - 404 errors properly fall back to local-only closure
 * 1.10.5.2 - 2025-08-03 (API Correction - Incorrect)
 *          - Fixed listing closure to use correct Delcampe API format
 *          - Changed from DELETE to POST method with XML body
 *          - Added proper Content-Type: application/xml header
 *          - Request body now includes Seller_Item_Close_UnSold structure
 *          - Closures are processed asynchronously via notifications
 * 1.10.5.1 - 2025-08-03 (Hotfix)
 *          - Added fallback to local-only listing closure when API returns HTML
 *          - Fixed "Failed to parse API response" error when closing listings
 *          - Created delcampe_mark_listing_ended() function for local updates
 *          - Added 'delcampe_use_local_close_only' filter for configuration
 *          - Users can now mark listings as ended locally even if API doesn't support it
 *          - Shows informative message to users about manual closure on Delcampe
 * 1.10.5.0 - Added listing closure functionality and fixed network issues
 *          - Implemented close/end listing feature in plugin interface
 *          - Added image proxy system for complex network setups
 *          - Fixed redirect loop issues preventing image access
 * 1.10.4.0 - Image proxy and network diagnostics (intermediate)
 * 1.10.3.0 - Enhanced API response handling for 201 Created status
 *          - Full support for new Delcampe API response format
 *          - Proper handling of Seller_Item_Add notification type
 *          - Support for warning messages in successful responses
 *          - Updated sync logger and viewer for better diagnostics
 * 1.10.2.0 - Added closed listings synchronization and cleanup functionality
 *          - New "Sync Closed Listings" feature to sync items closed on Delcampe
 *          - Added "Cleanup All Listings" to check all listings against Delcampe
 *          - Fixed database column issues (status vs listing_status, updated_at)
 *          - Enhanced error handling for listings without Delcampe IDs
 *          - Improved Orders Admin interface with new sync options
 * 1.10.1.0 - Fixed Orders menu visibility and loading issues
 *          - Added Orders Admin class to main plugin file loading
 *          - Fixed menu registration for orders management
 * 1.10.0.0 - Major order management system update
 *          - Added closed sales sync functionality
 *          - Enhanced API integration for order status updates
 * 1.9.0.0 - Added comprehensive bulk publish functionality
 *         - New bulk publish interface with multiple selection methods
 *         - Support for publishing by category, tags, filters, or all products
 *         - Real-time progress tracking with batch processing
 *         - Preview products before publishing
 *         - Fixed sync queue auto-removal after successful sync
 *         - Enhanced sync handler debugging and reliability
 * 1.8.1.0 - Critical fixes for license system and WooCommerce compatibility
 *         - Fixed license redirect loop preventing menu access
 *         - Updated license server URL to production (sos-tech.ca)
 *         - Added full HPOS (High-Performance Order Storage) compatibility
 *         - Fixed undefined constant errors in license manager
 *         - Enhanced order meta box handling for both storage types
 * 1.8.0.0 - Production release with data isolation and clean installation support
 *         - Fixed data carryover issue - plugin now requires clean configuration
 *         - Added comprehensive license key system for feature access control
 *         - Added setup notice for first-time configuration
 *         - Removed all hardcoded data and API keys from plugin files
 *         - Enhanced security with proper data storage in WordPress database
 *         - Added .gitignore to prevent sensitive data inclusion
 *         - Comprehensive documentation throughout codebase
 *         - All user data now stored exclusively in database
 * 1.6.2.1 - Fixed duplicate inventory menu and missing render_page() method
 *         - Removed duplicate menu registration in inventory admin class
 *         - Added render_page() method for compatibility
 *         - Updated plugin header with correct author information
 * 1.6.2.0 - Fixed database schema to include personal_reference column
 *         - Added database migration for existing installations
 *         - Fixed sync handler to properly handle personal_reference field
 *         - Updated version numbers across all affected files
 *         - Added comprehensive comments explaining the fix
 * Earlier versions - See full changelog in CHANGELOG.md
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Load plugin constants file first - this centralizes all version information
require_once plugin_dir_path( __FILE__ ) . 'includes/plugin-constants.php';

// Register autoloader for Delcampe classes (fallback to manual requires remains)
$__delcampe_autoload = plugin_dir_path( __FILE__ ) . 'includes/autoload.php';
if ( file_exists( $__delcampe_autoload ) ) {
    require_once $__delcampe_autoload;
}

/**
 * Define plugin constants for use throughout the plugin
 * These constants provide easy access to plugin version, directories, and URLs
 * Now using centralized constants from plugin-constants.php where applicable
 * 
 * @since 1.0.0
 */
define( 'DWC_VERSION', DELCAMPE_PLUGIN_VERSION ); // Plugin version from constants file
define( 'DWC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) ); // Full path to plugin directory
define( 'DWC_PLUGIN_URL', plugin_dir_url( __FILE__ ) ); // URL to plugin directory
define( 'DWC_PLUGIN_BASENAME', plugin_basename( __FILE__ ) ); // Plugin basename for hooks
define( 'DELCAMPE_PATH', plugin_dir_path( __FILE__ ) ); // Add DELCAMPE_PATH for license system

/**
 * Load core classes required for plugin functionality
 * These are the essential components needed for basic plugin operation
 * NOTE: We only load the files here, not instantiate them to avoid early translation loading
 * 
 * @since 1.0.0
 */

// License system (v1.8.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/license/class-delcampe-license-manager.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/license/class-delcampe-license-manager.php';
}
if ( file_exists( DWC_PLUGIN_DIR . 'includes/license/delcampe-license-functions.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/license/delcampe-license-functions.php';
}

// Setup notice handler for clean installations (v1.8.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-setup-notice.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-setup-notice.php';
}

// Declare HPOS compatibility
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );

// Logging functions (v1.10.2.0)
require_once DWC_PLUGIN_DIR . 'includes/delcampe-logging-functions.php';

// Sync logger class for structured logging (v1.10.10.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-logger.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-logger.php';
}

// Upgrade handler to prevent file deletion during updates (v1.10.19.1)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-upgrader.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-upgrader.php';
}

// Authentication handler for Delcampe API token management
require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-auth.php';

// Category API wrapper for retrieving Delcampe categories
require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-category-api.php';

// Lightweight API client wrapper used by some admin templates
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-api-client.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-api-client.php';
}

// Admin menu loader for registering plugin pages in WordPress admin
require_once DWC_PLUGIN_DIR . 'admin/class-delcampe-admin-menu.php';

// Admin page renderer - needed for table creation and backward compatibility
require_once DWC_PLUGIN_DIR . 'admin/class-admin-page.php';

// Profiles model for category mapping profiles
require_once DWC_PLUGIN_DIR . 'includes/profiles/class-delcampe-profiles-model.php';

// Database upgrade handler for schema updates
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-db-upgrade.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-db-upgrade.php';
}

// Profile assignments migration (v1.10.34.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/migrations/fix-profile-assignments.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/migrations/fix-profile-assignments.php';
}

// Product integration for profile assignment (v1.2.0.0)
require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-product-integration.php';

// Listing API for creating listings on Delcampe (v1.2.1.0)
require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listing-api.php';

// Sync handler for queue-based processing (v1.2.1.0)
require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-handler.php';

// Table-backed FIFO sync queue (v2 scaffold)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-queue.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-queue.php';
}

// Batch queue manager for improved batch processing (v1.10.18.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-batch-queue.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-batch-queue.php';
}

// Batch queue monitor for admin interface (v1.10.18.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-batch-monitor.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-batch-monitor.php';
}

// Shipping models manager (v1.3.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-shipping-models.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-shipping-models.php';
}

// Order manager for order synchronization (v1.4.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-order-manager.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-order-manager.php';
}

// Order reconciliation system for proper dollar amount handling (v1.10.13.16)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-order-reconciliation.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-order-reconciliation.php';
}

// Stuck order resolver for automatic payment data recovery (v1.10.36.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-stuck-order-resolver.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-stuck-order-resolver.php';
    Delcampe_Stuck_Order_Resolver::get_instance();
}

// Shipping helper for WooCommerce-side shipping (v1.10.13.8)
// Since Delcampe API doesn't support shipping status, we handle it locally
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-shipping-helper.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-shipping-helper.php';
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-woo-integration.php';
}

// Orders admin interface (v1.4.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-orders-admin.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-orders-admin.php';
}

// Inventory manager for stock synchronization (v1.5.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-inventory-manager.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-inventory-manager.php';
}

// Plugin Update Checker (PUC) library (v1.10.9.0)
if ( file_exists( DWC_PLUGIN_DIR . 'lib/plugin-update-checker/plugin-update-checker.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'lib/plugin-update-checker/plugin-update-checker.php';
}

// Inventory admin interface (v1.5.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-inventory-admin.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-inventory-admin.php';
}

// Listings model for managing Delcampe listings (v1.6.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-listings-model.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listings-model.php';
}

// Listing tracker for audit logging and idempotency (v1.10.18.13)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-listing-tracker.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listing-tracker.php';
}

// Reconciliation system for daily sync verification (v1.10.18.13)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-reconciliation.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-reconciliation.php';
}

// Sync dashboard widget for admin visibility (v1.10.18.13)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-dashboard.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-sync-dashboard.php';
}

// Listings sync for syncing listing status (v1.10.2.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-listings-sync.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listings-sync.php';
}

// Webhook manager for programmatic webhook registration (v1.10.5.17)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-manager.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-manager.php';
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-guardian.php';
}

// Webhook registration interface (v1.10.13.21)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-registration.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-registration.php';
    Delcampe_Webhook_Registration::get_instance();
    
    // Initialize webhook guardian for production-safe management (v1.10.14.0)
    if ( class_exists( 'Delcampe_Webhook_Guardian' ) ) {
        Delcampe_Webhook_Guardian::get_instance();
    }
    
    // DEPRECATED: Old webhook registration - replaced by Webhook Guardian (v1.10.14.0)
    // The Guardian handles this more efficiently with proper URL normalization and locking
    /*
    if (!wp_next_scheduled('delcampe_ensure_webhooks')) {
        wp_schedule_event(time() + 300, 'twicedaily', 'delcampe_ensure_webhooks');
    }
    
    add_action('delcampe_ensure_webhooks', function() {
        if (class_exists('Delcampe_Webhook_Registration')) {
            Delcampe_Webhook_Registration::get_instance()->ensure_default_webhooks();
        }
    });
    
    add_action('admin_init', function() {
        static $run_once = false;
        if (!$run_once && class_exists('Delcampe_Webhook_Registration')) {
            $run_once = true;
            Delcampe_Webhook_Registration::get_instance()->ensure_default_webhooks();
        }
    });
    */
}

// Listings table for displaying listings (v1.6.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/tables/class-delcampe-listings-list-table.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/tables/class-delcampe-listings-list-table.php';
}

// Bulk publish functionality (v1.9.0.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-bulk-publish.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-bulk-publish.php';
}

// Listings admin interface (v1.6.0.0)
// Only load the file - instantiation is handled by admin menu
if ( file_exists( DWC_PLUGIN_DIR . 'admin/class-delcampe-listings-admin.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'admin/class-delcampe-listings-admin.php';
}

// Development utilities (v1.2.2.1)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-dev-utils.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-dev-utils.php';
}

// Image URL filter for dev environment (v1.10.4.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/filters/image-url-filter.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/filters/image-url-filter.php';
}

// Image proxy for complex network setups (v1.10.4.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-image-proxy.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-image-proxy.php';
}

// Listing closer for removing listings (v1.10.5.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-listing-closer.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listing-closer.php';
}

// Price sync for automatic price synchronization (v1.10.10.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-price-sync.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-price-sync.php';
}

// Image manager for import/export of images (v1.10.10.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-image-manager.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-image-manager.php';
}

// Enhanced bulk operations (v1.10.10.0)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-bulk-operations.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-bulk-operations.php';
}

// Deadlock prevention for Action Scheduler (v1.10.10.1)
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-deadlock-prevention.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-deadlock-prevention.php';
}

/**
 * Main Plugin Loader Class
 * 
 * Handles initialization of the plugin, loading required files,
 * and setting up activation hooks. Uses singleton pattern for efficiency.
 * 
 * @since   1.0.0
 * @version 1.8.0.0
 */
class DWC_Loader {
    
    /**
     * Plugin Update Checker instance
     * @var object
     */
    private static $update_checker = null;
    
    /**
     * Initialize the plugin
     * 
     * This is the main entry point for plugin initialization.
     * Called on 'plugins_loaded' action to ensure WordPress is ready.
     * Order of operations is critical to avoid translation loading issues.
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return void
     */
    public static function init() {
        // Initialize license manager first (v1.8.0.0)
        if ( class_exists( 'Delcampe_License_Manager' ) ) {
            Delcampe_License_Manager::get_instance();
        }
        
        // Check license before initializing other components
        if ( ! delcampe_has_valid_license() ) {
            // Only allow access to license page and basic admin functionality
            // Defer admin notices until after init to avoid early translation loading (v1.10.5.10)
            add_action( 'init', function() {
                add_action( 'admin_notices', array( 'DWC_Loader', 'show_license_required_notice' ) );
            } );
            // Schedule text domain loading at init hook for WordPress 6.7+ compatibility
            add_action( 'init', array( __CLASS__, 'load_textdomain' ) );
            if ( is_admin() ) {
                Delcampe_Admin_Menu::get_instance();
            }
            return; // Exit early if no valid license
        }
        
        // Schedule text domain loading at init hook for WordPress 6.7+ compatibility
        add_action( 'init', array( __CLASS__, 'load_textdomain' ) );
        
        // Load all core functionality files
        self::load_core_files();
        
        // If in admin area, initialize admin-specific functionality
        if ( is_admin() ) {
            self::init_admin();
        }
        
        // Initialize product integration (v1.2.0.0)
        if ( class_exists( 'Delcampe_Product_Integration' ) && delcampe_has_feature( 'sync' ) ) {
            Delcampe_Product_Integration::get_instance();
        }
        
        // Initialize listing API (v1.2.1.0)
        if ( class_exists( 'Delcampe_Listing_API' ) && delcampe_has_feature( 'listings' ) ) {
            Delcampe_Listing_API::get_instance();
        }
        
        // Initialize sync handler (v1.2.1.0)
        if ( class_exists( 'Delcampe_Sync_Handler' ) && delcampe_has_feature( 'sync' ) ) {
            Delcampe_Sync_Handler::get_instance();
        }
        
        // Initialize shipping models manager (v1.3.0.0)
        if ( class_exists( 'Delcampe_Shipping_Models' ) ) {
            Delcampe_Shipping_Models::get_instance();
        }
        
        // Initialize order manager (v1.4.0.0)
        if ( class_exists( 'Delcampe_Order_Manager' ) && delcampe_has_feature( 'orders' ) ) {
            Delcampe_Order_Manager::get_instance();
        }
        
        // Initialize inventory manager (v1.5.0.0)
        if ( class_exists( 'Delcampe_Inventory_Manager' ) && delcampe_has_feature( 'inventory' ) ) {
            Delcampe_Inventory_Manager::get_instance();
        }
        
        // Initialize inventory admin interface (v1.5.0.0)
        if ( class_exists( 'Delcampe_Inventory_Admin' ) && delcampe_has_feature( 'inventory' ) ) {
            Delcampe_Inventory_Admin::get_instance();
        }
        
        // Initialize duplicate monitor (v1.10.35.0)
        if ( class_exists( 'Delcampe_Duplicate_Monitor' ) ) {
            Delcampe_Duplicate_Monitor::get_instance();
        }
        
        // Initialize Plugin Update Checker (v1.10.9.0)
        // Using PUC library v5.3 by Yahnis Elsts for reliable WordPress updates
        if ( class_exists( '\YahnisElsts\PluginUpdateChecker\v5\PucFactory' ) ) {
            self::$update_checker = \YahnisElsts\PluginUpdateChecker\v5\PucFactory::buildUpdateChecker(
                'https://www.sos-tech.ca/plugin-updates/delcampe/info.json',
                __FILE__,
                'delcampe-sync'
            );
            
            // Clear stale update cache if version matches (v1.10.12.4)
            self::$update_checker->addFilter( 'request_info_result', function( $pluginInfo, $result ) {
                if ( $pluginInfo && isset( $pluginInfo->version ) ) {
                    // If remote version matches or is older than current, clear false positive
                    if ( version_compare( $pluginInfo->version, DELCAMPE_PLUGIN_VERSION, '<=' ) ) {
                        // Clear the update notification
                        $pluginInfo = null;
                    }
                }
                return $pluginInfo;
            }, 10, 2 );
        }
        
        // Initialize image proxy (v1.10.4.0)
        if ( class_exists( 'Delcampe_Image_Proxy' ) ) {
            Delcampe_Image_Proxy::get_instance();
        }

        // Register WP-CLI commands if available (v1.10.14.8)
        if ( defined('WP_CLI') && WP_CLI && file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-cli.php' ) ) {
            require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-cli.php';
        }
        
        // Initialize listing closer (v1.10.5.0)
        if ( class_exists( 'Delcampe_Listing_Closer' ) ) {
            Delcampe_Listing_Closer::get_instance();
        }
        
        // Initialize Master Sync Manager (v1.10.10.4)
        require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-master-sync.php';
        if ( class_exists( 'Delcampe_Master_Sync' ) && delcampe_has_feature( 'sync' ) ) {
            Delcampe_Master_Sync::get_instance();
        }

        // Register AJAX for requeue stuck listings (v1.10.15.6)
        if ( class_exists( 'Delcampe_Sync_Handler' ) ) {
            add_action('wp_ajax_delcampe_requeue_stuck_listings', array( Delcampe_Sync_Handler::get_instance(), 'ajax_requeue_stuck_listings' ));
            add_action('wp_ajax_delcampe_reconcile_open_listings', array( Delcampe_Sync_Handler::get_instance(), 'ajax_reconcile_open_listings' ));
            add_action('wp_ajax_delcampe_rebuild_local_state', array( Delcampe_Sync_Handler::get_instance(), 'ajax_rebuild_local_state' ));
        }
        
        // Initialize Business Event Logger (v1.10.12.0)
        require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-business-logger.php';
        if ( class_exists( 'Delcampe_Business_Logger' ) ) {
            Delcampe_Business_Logger::get_instance();
        }
        
        // Initialize Order Consolidation System (v1.10.27.0)
        if (file_exists(DWC_PLUGIN_DIR . 'includes/class-delcampe-order-consolidation.php')) {
            require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-order-consolidation.php';
            if (class_exists('Delcampe_Order_Consolidation')) {
                Delcampe_Order_Consolidation::get_instance();
            }
        }
        
        // Initialize Real-time Sync Handler (v1.10.12.0)
        require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-realtime-sync.php';
        if ( class_exists( 'Delcampe_Realtime_Sync' ) && delcampe_has_feature( 'sync' ) ) {
            Delcampe_Realtime_Sync::get_instance();
        }
        
        // Initialize Business Log Admin Interface (v1.10.12.0)
        if ( is_admin() ) {
            require_once DWC_PLUGIN_DIR . 'admin/class-delcampe-business-log-admin.php';
            if ( class_exists( 'Delcampe_Business_Log_Admin' ) ) {
                Delcampe_Business_Log_Admin::get_instance();
            }
        }
        
        // Initialize New Queue System (v2.0.0)
        // This replaces the old complex sync handler and batch queue systems
if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-queue-system.php' ) ) {
    require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-queue-system.php';
    if ( class_exists( 'Delcampe_Queue_System' ) && delcampe_has_feature( 'sync' ) ) {
        Delcampe_Queue_System::get_instance();
    }
}

// Bridge old sync queue hook to new queue worker for manual runs/buttons
add_action('delcampe_process_sync_queue', function() {
    if ( class_exists('Delcampe_Queue_Worker') ) {
        // Process all items, not just 25
        Delcampe_Queue_Worker::get_instance()->process_batch(9999);
    }
});
        
        // Note: Listings admin (v1.6.0.1) is now initialized by Delcampe_Admin_Menu
        // to ensure proper menu callback handling
    }
    
    /**
     * Show license required notice
     * 
     * @since 1.8.0.0
     * @return void
     */
    public static function show_license_required_notice() {
        $screen = get_current_screen();
        // Don't show on license page
        if ( $screen && $screen->id === 'delcampe_page_delcampe-license' ) {
            return;
        }
        ?>
        <div class="notice notice-error">
            <p>
                <?php
                printf(
                    /* translators: %s: Link to license page */
                    esc_html__( 'WooCommerce Delcampe Integration requires a valid license key. %s', 'wc-delcampe-integration' ),
                    '<a href="' . esc_url( admin_url( 'admin.php?page=delcampe-license' ) ) . '">' . esc_html__( 'Enter your license key', 'wc-delcampe-integration' ) . '</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }

    /**
     * Check for plugin updates manually
     * 
     * Triggers the Plugin Update Checker to look for new versions
     * 
     * @since 1.10.9.1
     * @return void
     */
    public static function check_for_updates() {
        // Clear all update-related caches first (v1.10.12.4)
        delete_site_transient( 'update_plugins' );
        delete_transient( 'delcampe_update_info' );
        delete_transient( 'puc_update_check-delcampe-sync' );
        delete_option( '_site_transient_update_plugins' );
        wp_cache_delete( 'plugins', 'transient' );
        
        if ( self::$update_checker ) {
            // Reset the update checker state
            self::$update_checker->resetUpdateState();
            
            // Force check for updates
            $update = self::$update_checker->checkForUpdates();
            
            // If no update or current version is newer/same, ensure cache is clear
            if ( !$update || version_compare( $update->version, DELCAMPE_PLUGIN_VERSION, '<=' ) ) {
                // Make sure WordPress knows there's no update
                $current = get_site_transient( 'update_plugins' );
                if ( $current && isset( $current->response['delcampe-sync/main.php'] ) ) {
                    unset( $current->response['delcampe-sync/main.php'] );
                    set_site_transient( 'update_plugins', $current );
                }
            }
        }
        
        // Redirect to plugins page to show update status
        wp_safe_redirect( admin_url( 'plugins.php' ) );
        exit;
    }
    
    /**
     * Load plugin text domain for translations
     * 
     * Must be called at init or later to avoid WordPress 6.7 warnings.
     * Now properly hooked to 'init' action for WordPress 6.7+ compatibility.
     * Uses constants from plugin-constants.php for consistency.
     * 
     * @since  1.1.1.0
     * @version 1.9.0.0
     * @return void
     */
    public static function load_textdomain() {
        load_plugin_textdomain(
            DELCAMPE_TEXT_DOMAIN,
            false,
            dirname( plugin_basename( __FILE__ ) ) . DELCAMPE_DOMAIN_PATH
        );
    }

    /**
     * Load core plugin files
     * 
     * Loads additional functionality files that aren't required immediately
     * but are needed for full plugin operation. Files are only loaded if they exist.
     * This prevents fatal errors if files are missing during development.
     * Enhanced in v1.1.4.0 to include stamp-specific functionality.
     * Enhanced in v1.3.0.0 to include shipping admin interface.
     * 
     * @since  1.0.0
     * @version 1.3.0.0
     * @return void
     */
    private static function load_core_files() {
        // Array of files to load relative to plugin directory
        $core_files = array(
            'includes/class-delcampe-category-mapping.php',    // Handles category mapping between platforms
            'includes/class-delcampe-category-manager.php',    // Manages category hierarchies and caching
            'admin/class-delcampe-category-admin.php',         // Admin interface for category management
            'includes/class-delcampe-test-sync.php',           // Test synchronization functionality (optional - for development)
            'includes/profiles/stamp-profile-fields.php',      // Stamp-specific field definitions (v1.1.4.0)
            'includes/profiles/class-delcampe-description-builder.php', // Description builder (v1.1.4.0)
            'includes/class-delcampe-duplicate-monitor.php',   // Duplicate prevention monitoring (v1.10.35.0)
        );
        
        // Iterate through files and load if they exist
        foreach ( $core_files as $file ) {
            $path = DWC_PLUGIN_DIR . $file;
            if ( file_exists( $path ) ) {
                require_once $path;
            }
        }
    }

    /**
     * Initialize admin-specific functionality
     * 
     * Sets up admin-only features and hooks.
     * Now properly initializes admin menu and tabbed interface after translations are loaded.
     * This prevents "translation called too early" warnings in WordPress 6.7+
     * Enhanced in v1.3.0.0 to include shipping admin interface.
     * 
     * @since  1.0.0
     * @version 1.3.0.0
     * @return void
     */
    private static function init_admin() {
        // Initialize the admin menu (delayed to avoid early translation loading)
        // This now handles initialization of all admin modules including listings
        Delcampe_Admin_Menu::get_instance();
        
        // Initialize the admin page functionality for backward compatibility
        if ( class_exists( 'Delcampe_Admin_Page' ) ) {
            Delcampe_Admin_Page::init();
        }
        
        // Initialize tabbed admin interface AFTER translations are loaded
        if ( class_exists( 'Delcampe_Admin_Tabbed_Page' ) ) {
            Delcampe_Admin_Tabbed_Page::init();
        }
        
        // Initialize rate limit monitoring (v1.10.12.5)
        // This adds rate limit display to all Delcampe admin pages
        require_once plugin_dir_path( __FILE__ ) . 'includes/class-delcampe-rate-limit.php';
        if ( class_exists( 'Delcampe_Rate_Limit' ) ) {
            Delcampe_Rate_Limit::get_instance();
        }
    }

    /**
     * Plugin activation routine
     * 
     * Runs when the plugin is activated. Sets up database tables
     * and flushes rewrite rules to ensure proper URL handling.
     * Enhanced in version 1.8.0.0 to ensure clean installations and license table.
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return void
     */
    public static function activate() {
        // Log activation event
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Activation hook triggered' );
        
        // Clear any stale update caches (v1.10.12.4)
        delete_site_transient( 'update_plugins' );
        delete_transient( 'puc_update_check-delcampe-sync' );
        delete_option( '_site_transient_update_plugins' );
        wp_cache_delete( 'plugins', 'transient' );
        
        // v1.10.26.1 - Clear any stuck locks on activation/upgrade
        self::clear_stuck_locks();
        
        // Run version-specific upgrades
        self::run_version_upgrades();
        
        // Check minimum requirements
        if ( version_compare( PHP_VERSION, DELCAMPE_MIN_PHP, '<' ) ) {
            deactivate_plugins( plugin_basename( __FILE__ ) );
            wp_die( sprintf(
                'This plugin requires PHP version %s or higher.',
                DELCAMPE_MIN_PHP
            ) );
        }
        
        // Check if WooCommerce is active
        if ( ! class_exists( 'WooCommerce' ) ) {
            deactivate_plugins( plugin_basename( __FILE__ ) );
            wp_die( 'This plugin requires WooCommerce to be installed and active.' );
        }
        
        // Create license table (v1.8.0.0)
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        $license_table = $wpdb->prefix . 'delcampe_license';
        $sql = "CREATE TABLE IF NOT EXISTS {$license_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            license_key varchar(255) NOT NULL,
            license_status varchar(50) DEFAULT 'inactive',
            license_type varchar(50) DEFAULT 'basic',
            activation_date datetime DEFAULT NULL,
            expiration_date datetime DEFAULT NULL,
            activation_limit int(11) DEFAULT 1,
            activation_count int(11) DEFAULT 0,
            instance_id varchar(255) DEFAULT NULL,
            last_check datetime DEFAULT NULL,
            cached_response longtext,
            PRIMARY KEY (id),
            UNIQUE KEY license_key (license_key)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] License table created' );
        
        // Create category mapping table if the class is available
        if ( class_exists( 'Delcampe_Category_Mapping' ) ) {
            require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-category-mapping.php';
            $mapping_instance = Delcampe_Category_Mapping::get_instance();
            $mapping_instance->create_mapping_table();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Category mapping table created' );
        }
        
        // Create admin page table - this is the one we need for selected categories
        // Fixed in v1.1.2.1: Call the static method directly without relying on translations
        if ( method_exists( 'Delcampe_Admin_Page', 'create_category_table' ) ) {
            Delcampe_Admin_Page::create_category_table();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Selected categories table created' );
        }
        
        // Create profiles table
        if ( class_exists( 'Delcampe_Profiles_Model' ) ) {
            $profiles_model = Delcampe_Profiles_Model::get_instance();
            $profiles_model->create_table();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Profiles table created' );
        }
        
        // Create order management tables (v1.4.0.0)
        if ( class_exists( 'Delcampe_Order_Manager' ) ) {
            $order_manager = Delcampe_Order_Manager::get_instance();
            $order_manager->create_tables();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Order management tables created' );
        }
        
        // Create inventory management tables (v1.5.0.0)
        if ( class_exists( 'Delcampe_Inventory_Manager' ) ) {
            $inventory_manager = Delcampe_Inventory_Manager::get_instance();
            $inventory_manager->create_tables();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Inventory management tables created' );
        }
        
        // Create listings table (v1.6.0.0)
        // Updated in v1.6.2.0 to ensure personal_reference column is included
        // Updated in v1.10.35.0 to add duplicate prevention constraints
        if ( class_exists( 'Delcampe_Listings_Model' ) ) {
            Delcampe_Listings_Model::create_table();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Listings table created with personal_reference column' );
            
            // Add duplicate prevention constraints (v1.10.35.0)
            self::add_duplicate_prevention_constraints();
        }
        
        // Create Queue System v2.0 tables (v1.10.29.2)
        // CRITICAL: Must create tables on activation for production sites with custom prefixes
        if ( file_exists( DWC_PLUGIN_DIR . 'includes/class-delcampe-queue.php' ) ) {
            require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-queue.php';
            if ( class_exists( 'Delcampe_Queue' ) ) {
                $queue = Delcampe_Queue::get_instance();
                if ( method_exists( $queue, 'create_tables' ) ) {
                    $queue->create_tables();
                    delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Queue v2.0 tables created' );
                }
            }
        }
        
        // Create duplicate monitor table (v1.10.35.0)
        if ( class_exists( 'Delcampe_Duplicate_Monitor' ) ) {
            Delcampe_Duplicate_Monitor::create_table();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Duplicate monitor table created' );
        }
        
        // Run database upgrades if upgrade class exists
        if ( class_exists( 'Delcampe_DB_Upgrade' ) ) {
            Delcampe_DB_Upgrade::run_upgrades();
            delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Database upgrades completed' );
        }
        
        // Set default options using constants
        add_option( DELCAMPE_VERSION_OPTION, DELCAMPE_PLUGIN_VERSION );
        add_option( DELCAMPE_LOGGING_OPTION, false );
        add_option( DELCAMPE_AUTO_SYNC_INVENTORY_OPTION, true );
        add_option( DELCAMPE_INVENTORY_CHECK_FREQUENCY_OPTION, DELCAMPE_DEFAULT_INVENTORY_CHECK_INTERVAL );
        add_option( DELCAMPE_LOW_STOCK_ALERT_OPTION, DELCAMPE_DEFAULT_LOW_STOCK_THRESHOLD );
        add_option( DELCAMPE_PREVENT_OVERSELL_OPTION, true );
        add_option( DELCAMPE_ORDER_IMPORT_ENABLED_OPTION, true );
        add_option( DELCAMPE_ORDER_IMPORT_FREQUENCY_OPTION, 'hourly' );
        
        // Reset setup notice for clean installations (v1.8.0.0)
        delete_option( 'delcampe_setup_notice_dismissed' );
        
        // Add Action Scheduler indexes for better performance (v1.10.10.1)
        if ( class_exists( 'Delcampe_Deadlock_Prevention' ) ) {
            Delcampe_Deadlock_Prevention::add_indexes();
        }
        
        // Flush rewrite rules to ensure any custom URLs work properly
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation routine
     * 
     * Cleanup tasks when plugin is deactivated.
     * Clears scheduled tasks and flushes rewrite rules.
     * Enhanced in v1.8.0.0 to reset setup notice and clear license schedules.
     * 
     * @since  1.0.9.0
     * @version 1.8.0.0
     * @return void
     */
    public static function deactivate() {
        // Clear all scheduled cron tasks
        wp_clear_scheduled_hook( DELCAMPE_SYNC_CRON_HOOK );
        wp_clear_scheduled_hook( DELCAMPE_PROCESS_QUEUE_CRON_HOOK );
        wp_clear_scheduled_hook( DELCAMPE_IMPORT_ORDERS_CRON_HOOK );
        wp_clear_scheduled_hook( DELCAMPE_INVENTORY_CHECK_CRON_HOOK );
        wp_clear_scheduled_hook( 'delcampe_daily_license_check' );
        
        // Clear any transient data
        delete_transient( DELCAMPE_TOKEN_TRANSIENT );
        delete_transient( DELCAMPE_CATEGORIES_TRANSIENT );
        delete_transient( DELCAMPE_MAPPING_CACHE_KEY );
        delete_transient( DELCAMPE_PROFILES_CACHE_KEY );
        delete_transient( DELCAMPE_SHIPPING_MODELS_CACHE_KEY );
        delete_transient( 'delcampe_license_data' );
        
        // Clear stock cache cron (v1.10.20.3)
        wp_clear_scheduled_hook( 'delcampe_refresh_stock_cache' );
        
        // Reset setup notice (v1.8.0.0)
        if ( class_exists( 'Delcampe_Setup_Notice' ) ) {
            Delcampe_Setup_Notice::reset_notice();
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Log deactivation event
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Plugin deactivated' );
    }
    
    /**
     * Add duplicate prevention database constraints
     * 
     * Adds UNIQUE constraint on delcampe_id and compound index for fast lookups
     * Part of Phase 1 duplicate prevention (v1.10.35.0)
     * 
     * @since 1.10.35.0
     * @return void
     */
    private static function add_duplicate_prevention_constraints() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        // 1. Clean up any existing duplicates before adding constraint
        $wpdb->query("
            DELETE l1 FROM $table_name l1
            INNER JOIN $table_name l2 
            WHERE l1.id > l2.id 
            AND l1.delcampe_id = l2.delcampe_id 
            AND l1.delcampe_id IS NOT NULL
            AND l1.delcampe_id != ''
        ");
        
        // 2. Add UNIQUE constraint on delcampe_id if it doesn't exist
        $unique_exists = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS tc
            JOIN INFORMATION_SCHEMA.KEY_COLUMN_USAGE kcu 
                ON tc.CONSTRAINT_NAME = kcu.CONSTRAINT_NAME
            WHERE tc.TABLE_SCHEMA = DATABASE()
            AND tc.TABLE_NAME = '$table_name'
            AND tc.CONSTRAINT_TYPE = 'UNIQUE'
            AND kcu.COLUMN_NAME = 'delcampe_id'
        ");
        
        if (!$unique_exists) {
            // Use ALTER IGNORE to skip duplicates if any remain
            $wpdb->query("ALTER IGNORE TABLE $table_name ADD UNIQUE KEY unique_delcampe_id (delcampe_id)");
            delcampe_log('[Duplicate Prevention] Added UNIQUE constraint on delcampe_id');
        }
        
        // 3. Add compound index for fast duplicate checking
        $compound_exists = $wpdb->get_var("
            SELECT COUNT(*)
            FROM INFORMATION_SCHEMA.STATISTICS
            WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME = '$table_name'
            AND INDEX_NAME = 'idx_product_status'
        ");
        
        if (!$compound_exists) {
            $wpdb->query("ALTER TABLE $table_name ADD INDEX idx_product_status (product_id, status)");
            delcampe_log('[Duplicate Prevention] Added compound index on (product_id, status)');
        }
        
        // 4. Clean up orphaned listings (product doesn't exist)
        $wpdb->query("
            DELETE l FROM $table_name l
            LEFT JOIN {$wpdb->posts} p ON l.product_id = p.ID
            WHERE p.ID IS NULL
        ");
        
        delcampe_log('[Duplicate Prevention] Database constraints configured successfully');
    }
    
    /**
     * Clear stuck locks and fix common issues
     * Runs automatically on activation/upgrade
     * 
     * @since 1.10.26.1
     */
    public static function clear_stuck_locks() {
        // Clear master sync lock
        delete_transient('delcampe_master_sync_lock');
        
        // Clear batch processing pause
        delete_transient('delcampe_batch_processing_paused');
        
        // Reset stuck batches
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_batch_queue';
        
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name) {
            // Reset batches stuck in processing for over 10 minutes
            $wpdb->query("
                UPDATE {$table_name}
                SET status = 'queued',
                    processing_started = NULL,
                    last_error = 'Reset by upgrade routine v1.10.26.1'
                WHERE status = 'processing'
                AND processing_started < DATE_SUB(NOW(), INTERVAL 10 MINUTE)
            ");
        }
        
        delcampe_log('[Upgrade] Cleared stuck locks and reset frozen batches');
    }
    
    /**
     * Run version-specific upgrade routines
     * 
     * @since 1.10.26.1
     */
    public static function run_version_upgrades() {
        $current_version = get_option('delcampe_plugin_version', '0.0.0');
        
        // v1.10.26.1 - Fix alert formats
        if (version_compare($current_version, '1.10.26.1', '<')) {
            global $wpdb;
            $alerts_table = $wpdb->prefix . 'delcampe_inventory_alerts';
            
            if ($wpdb->get_var("SHOW TABLES LIKE '{$alerts_table}'") === $alerts_table) {
                // Update old alert formats
                $wpdb->query("
                    UPDATE {$alerts_table}
                    SET alert_type = 'sync_needed',
                        alert_message = REPLACE(alert_message, 'Inventory mismatch:', 'Delcampe out of sync:')
                    WHERE alert_type = 'sync_error'
                    AND is_resolved = 0
                ");
                
                delcampe_log('[Upgrade 1.10.26.1] Updated alert formats');
            }
        }
        
        // v1.10.35.0 - Add duplicate prevention constraints
        if (version_compare($current_version, '1.10.35.0', '<')) {
            // Add database constraints for duplicate prevention
            self::add_duplicate_prevention_constraints();
            
            // Create duplicate monitor table if it doesn't exist
            if ( class_exists( 'Delcampe_Duplicate_Monitor' ) ) {
                Delcampe_Duplicate_Monitor::create_table();
                delcampe_log('[Upgrade 1.10.35.0] Duplicate prevention system initialized');
            }
        }
        
        // Update stored version
        update_option('delcampe_plugin_version', DELCAMPE_PLUGIN_VERSION);
    }
}

/**
 * Register activation hook OUTSIDE of any class or function
 * This ensures it's registered immediately when the plugin file is loaded
 * 
 * @since 1.0.0
 */
register_activation_hook( __FILE__, array( 'DWC_Loader', 'activate' ) );

/**
 * Register deactivation hook
 * 
 * @since 1.0.9.0
 */
register_deactivation_hook( __FILE__, array( 'DWC_Loader', 'deactivate' ) );

/**
 * Hook into WordPress 'plugins_loaded' action
 * This ensures all plugins are loaded before we initialize
 * Priority 0 ensures early loading
 * 
 * @since 1.0.0
 */
add_action( 'plugins_loaded', array( 'DWC_Loader', 'init' ), 0 );

// Hook into upgrade process (v1.10.26.1)
// This ensures cleanup runs even on auto-updates
add_action( 'upgrader_process_complete', function( $upgrader, $options ) {
    if ( $options['action'] == 'update' && $options['type'] == 'plugin' ) {
        foreach( $options['plugins'] as $plugin ) {
            if ( $plugin == plugin_basename( __FILE__ ) ) {
                DWC_Loader::clear_stuck_locks();
                DWC_Loader::run_version_upgrades();
                delcampe_log('[Auto-Update] Ran cleanup routines after upgrade');
            }
        }
    }
}, 10, 2 );

/**
 * Display admin notice if WooCommerce is not active
 * 
 * This plugin requires WooCommerce to function properly.
 * Shows a warning in the admin area if WooCommerce is not detected.
 * 
 * @since  1.0.0
 * @version 1.8.0.0
 * @return void
 */
function delcampe_check_woocommerce_dependency() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        ?>
        <div class="error notice">
            <p>
                <?php
                printf(
                    /* translators: %s: Plugin name */
                    esc_html__( '%s requires WooCommerce to be installed and active.', 'wc-delcampe-integration' ),
                    '<strong>' . esc_html( DELCAMPE_PLUGIN_NAME ) . '</strong>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}

// Register the WooCommerce check after init to avoid early translation loading (v1.10.5.10)
add_action( 'init', function() {
    add_action( 'admin_notices', 'delcampe_check_woocommerce_dependency' );
} );

/**
 * Manual table creation function
 * 
 * Can be called to create tables if they don't exist.
 * This ensures tables are created even if activation hook didn't run properly.
 * Enhanced in version 1.8.0.0 for clean installations and license table.
 * 
 * @since  1.0.0
 * @version 1.8.0.0
 * @return void
 */
function delcampe_create_tables_if_needed() {
    global $wpdb;
    
    // Check license table (v1.8.0.0)
    $license_table = $wpdb->prefix . 'delcampe_license';
    $license_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $license_table ) ) === $license_table;
    
    if ( ! $license_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] License table does not exist, creating it now' );
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$license_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            license_key varchar(255) NOT NULL,
            license_status varchar(50) DEFAULT 'inactive',
            license_type varchar(50) DEFAULT 'basic',
            activation_date datetime DEFAULT NULL,
            expiration_date datetime DEFAULT NULL,
            activation_limit int(11) DEFAULT 1,
            activation_count int(11) DEFAULT 0,
            instance_id varchar(255) DEFAULT NULL,
            last_check datetime DEFAULT NULL,
            cached_response longtext,
            PRIMARY KEY (id),
            UNIQUE KEY license_key (license_key)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
    }
    
    // Check selected categories table
    $table_name = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
    $table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) === $table_name;
    
    if ( ! $table_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Selected categories table does not exist, creating it now' );
        if ( class_exists( 'Delcampe_Admin_Page' ) ) {
            Delcampe_Admin_Page::create_category_table();
        }
    }
    
    // Check profiles table
    $profiles_table = $wpdb->prefix . DELCAMPE_TABLE_PROFILES;
    $profiles_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $profiles_table ) ) === $profiles_table;
    
    if ( ! $profiles_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Profiles table does not exist, creating it now' );
        if ( class_exists( 'Delcampe_Profiles_Model' ) ) {
            $profiles_model = Delcampe_Profiles_Model::get_instance();
            $profiles_model->create_table();
        }
    }
    
    // Check order tables (v1.4.0.0)
    $orders_table = $wpdb->prefix . DELCAMPE_TABLE_ORDERS;
    $orders_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $orders_table ) ) === $orders_table;
    
    if ( ! $orders_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Order tables do not exist, creating them now' );
        if ( class_exists( 'Delcampe_Order_Manager' ) ) {
            $order_manager = Delcampe_Order_Manager::get_instance();
            $order_manager->create_tables();
        }
    }
    
    // Check inventory tables (v1.5.0.0)
    $inventory_table = $wpdb->prefix . DELCAMPE_TABLE_INVENTORY_SYNC;
    $inventory_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $inventory_table ) ) === $inventory_table;
    
    if ( ! $inventory_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Inventory tables do not exist, creating them now' );
        if ( class_exists( 'Delcampe_Inventory_Manager' ) ) {
            $inventory_manager = Delcampe_Inventory_Manager::get_instance();
            $inventory_manager->create_tables();
        }
    }
    
    // Check listings table (v1.6.0.0)
    // Updated in v1.6.2.0 to always call create_table which now handles column checks
    $listings_table = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;
    $listings_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $listings_table ) ) === $listings_table;
    
    if ( ! $listings_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Listings table does not exist, creating it now' );
        if ( class_exists( 'Delcampe_Listings_Model' ) ) {
            Delcampe_Listings_Model::create_table();
        }
    } else {
        // Table exists, but ensure personal_reference column is present (v1.6.2.0)
        if ( class_exists( 'Delcampe_Listings_Model' ) ) {
            Delcampe_Listings_Model::create_table(); // This now checks for and adds missing columns
        }
    }
    
    // v1.10.35.6: Check Queue v2.0 tables
    $queue_table = $wpdb->prefix . 'delcampe_queue';
    $queue_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $queue_table ) ) === $queue_table;
    
    if ( ! $queue_exists ) {
        delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Queue v2.0 table does not exist, creating it now' );
        if ( class_exists( 'Delcampe_Queue' ) ) {
            $queue = Delcampe_Queue::get_instance();
            if ( method_exists( $queue, 'create_tables' ) ) {
                $queue->create_tables();
                delcampe_log( '[Delcampe Plugin v' . DELCAMPE_PLUGIN_VERSION . '] Queue v2.0 tables created successfully' );
            }
        }
    }
}

// Run table creation on admin_init to ensure tables exist
add_action( 'admin_init', 'delcampe_create_tables_if_needed' );

/**
 * Handle manual update check action
 * 
 * Processes the request when user clicks "Check for updates" link
 * 
 * @since 1.10.9.1
 * @return void
 */
function delcampe_handle_check_updates() {
    // Check for our action
    if ( isset( $_GET['delcampe_check_updates'] ) && $_GET['delcampe_check_updates'] === '1' ) {
        // Verify nonce for security
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'delcampe_check_updates' ) ) {
            wp_die( __( 'Security check failed', 'wc-delcampe-integration' ) );
        }
        
        // Check user permissions
        if ( ! current_user_can( 'update_plugins' ) ) {
            wp_die( __( 'You do not have permission to update plugins', 'wc-delcampe-integration' ) );
        }
        
        // Simply trigger the update check using DWC_Loader's method
        // This avoids creating duplicate update checker instances
        if ( method_exists( 'DWC_Loader', 'check_for_updates' ) ) {
            DWC_Loader::check_for_updates();
        } else {
            // Fallback: Just clear transients and redirect
            // This will trigger WordPress to check all plugins for updates
            delete_site_transient( 'update_plugins' );
            delete_transient( 'delcampe_update_info' );
            
            // Redirect to plugins page
            wp_safe_redirect( admin_url( 'plugins.php' ) );
            exit;
        }
    }
}
add_action( 'admin_init', 'delcampe_handle_check_updates', 5 ); // Run early

/**
 * Check and update plugin version
 * 
 * Compares stored version with current version and runs upgrades if needed.
 * This ensures proper migration between plugin versions.
 * 
 * @since  1.0.9.0
 * @version 1.8.0.0
 * @return void
 */
function delcampe_check_version() {
    $current_version = get_option( DELCAMPE_VERSION_OPTION, '0' );
    
    if ( version_compare( $current_version, DELCAMPE_PLUGIN_VERSION, '<' ) ) {
        // Log version update
        delcampe_log( '[Delcampe Plugin] Updating from version ' . $current_version . ' to ' . DELCAMPE_PLUGIN_VERSION );
        
        // Version 1.3.0 upgrade: Add shipping fields to profiles table
        if ( version_compare( $current_version, '1.3.0', '<' ) ) {
            global $wpdb;
            $table_name = $wpdb->prefix . DELCAMPE_TABLE_PROFILES;
            
            // Check if shipping columns exist
            // SECURITY FIX v1.10.21.2: Properly escape table name
            $safe_table_check = esc_sql($table_name);
            $columns = $wpdb->get_results( "SHOW COLUMNS FROM `$safe_table_check`" );
            $column_names = array_map( function( $col ) { return $col->Field; }, $columns );
            
            // SECURITY FIX v1.10.21.2: Properly escape table names in schema operations
            $safe_table = esc_sql($table_name);
            
            // Add shipping_model_id column if it doesn't exist
            if ( ! in_array( 'shipping_model_id', $column_names, true ) ) {
                $wpdb->query( "ALTER TABLE `$safe_table` ADD COLUMN shipping_model_id INT DEFAULT 0 AFTER stamp_fields" );
                delcampe_log( '[Delcampe Plugin] Added shipping_model_id column to profiles table' );
            }
            
            // Add shipping_settings column if it doesn't exist
            if ( ! in_array( 'shipping_settings', $column_names, true ) ) {
                $wpdb->query( "ALTER TABLE `$safe_table` ADD COLUMN shipping_settings TEXT AFTER shipping_model_id" );
                delcampe_log( '[Delcampe Plugin] Added shipping_settings column to profiles table' );
            }
        }
        
        // Version 1.4.0 upgrade: Create order tables if they don't exist
        if ( version_compare( $current_version, '1.4.0', '<' ) ) {
            if ( class_exists( 'Delcampe_Order_Manager' ) ) {
                $order_manager = Delcampe_Order_Manager::get_instance();
                $order_manager->create_tables();
                delcampe_log( '[Delcampe Plugin] Created order management tables for v1.4.0' );
            }
        }
        
        // Version 1.5.0 upgrade: Create inventory tables if they don't exist
        if ( version_compare( $current_version, '1.5.0', '<' ) ) {
            if ( class_exists( 'Delcampe_Inventory_Manager' ) ) {
                $inventory_manager = Delcampe_Inventory_Manager::get_instance();
                $inventory_manager->create_tables();
                delcampe_log( '[Delcampe Plugin] Created inventory management tables for v1.5.0' );
            }
        }
        
        // Version 1.6.0 upgrade: Create listings table if it doesn't exist
        if ( version_compare( $current_version, '1.6.0', '<' ) ) {
            if ( class_exists( 'Delcampe_Listings_Model' ) ) {
                Delcampe_Listings_Model::create_table();
                delcampe_log( '[Delcampe Plugin] Created listings table for v1.6.0' );
            }
        }
        
        // Version 1.6.2.0 upgrade: Ensure personal_reference column exists
        if ( version_compare( $current_version, '1.6.2', '<' ) ) {
            if ( class_exists( 'Delcampe_Listings_Model' ) ) {
                // The create_table method now handles adding missing columns
                Delcampe_Listings_Model::create_table();
                delcampe_log( '[Delcampe Plugin] Ensured personal_reference column exists for v1.6.2.0' );
            }
        }
        
        // Version 1.8.0.0 upgrade: Reset setup notice for configuration and create license table
        if ( version_compare( $current_version, '1.8.0', '<' ) ) {
            delete_option( 'delcampe_setup_notice_dismissed' );
            
            // Create license table
            global $wpdb;
            $charset_collate = $wpdb->get_charset_collate();
            
            $license_table = $wpdb->prefix . 'delcampe_license';
            $sql = "CREATE TABLE IF NOT EXISTS {$license_table} (
                id int(11) NOT NULL AUTO_INCREMENT,
                license_key varchar(255) NOT NULL,
                license_status varchar(50) DEFAULT 'inactive',
                license_type varchar(50) DEFAULT 'basic',
                activation_date datetime DEFAULT NULL,
                expiration_date datetime DEFAULT NULL,
                activation_limit int(11) DEFAULT 1,
                activation_count int(11) DEFAULT 0,
                instance_id varchar(255) DEFAULT NULL,
                last_check datetime DEFAULT NULL,
                cached_response longtext,
                PRIMARY KEY (id),
                UNIQUE KEY license_key (license_key)
            ) $charset_collate;";
            
            require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
            dbDelta( $sql );
            
            delcampe_log( '[Delcampe Plugin] Reset setup notice and created license table for v1.8.0.0' );
        }
        
        // Version 1.9.0.3 upgrade: Add shipping_model_id column to profiles table
        if ( version_compare( $current_version, '1.9.0.3', '<' ) ) {
            if ( class_exists( 'Delcampe_Profiles_Model' ) ) {
                $profiles_model = Delcampe_Profiles_Model::get_instance();
                $profiles_model->create_table(); // This will add the missing column
                delcampe_log( '[Delcampe Plugin] Added shipping_model_id column for v1.9.0.3' );
            }
        }
        
        // Version 1.10.22.0 upgrade: Migrate license keys to encrypted format
        if ( version_compare( $current_version, '1.10.22.0', '<' ) ) {
            if ( class_exists( 'Delcampe_License_Manager' ) ) {
                $license_manager = Delcampe_License_Manager::get_instance();
                $migration_results = $license_manager->migrate_license_keys_to_encrypted();
                delcampe_log( '[Delcampe Plugin] License key encryption migration for v1.10.22.0 - ' . 
                             'Migrated: ' . $migration_results['migrated'] . ', ' .
                             'Already encrypted: ' . $migration_results['already_encrypted'] . ', ' .
                             'Failed: ' . $migration_results['failed'] );
            }
        }
        
        // Update stored version
        update_option( DELCAMPE_VERSION_OPTION, DELCAMPE_PLUGIN_VERSION );
    }
}
add_action( 'admin_init', 'delcampe_check_version' );

/**
 * Add plugin action links
 * Adds Settings and Support links to the plugin listing
 * 
 * @since  1.6.2.1
 * @param  array $links Existing plugin action links
 * @return array Modified action links
 */
function delcampe_plugin_action_links( $links ) {
    // Use hardcoded strings if text domain is not yet loaded
    $settings_text = 'Settings';
    $license_text = 'License';
    
    // Check if text domain is loaded, then translate
    if ( is_textdomain_loaded( 'wc-delcampe-integration' ) ) {
        $settings_text = __( 'Settings', 'wc-delcampe-integration' );
        $license_text = __( 'License', 'wc-delcampe-integration' );
    }
    
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_SETTINGS ) ),
        esc_html( $settings_text )
    );
    
    $license_link = sprintf(
        '<a href="%s">%s</a>',
        esc_url( admin_url( 'admin.php?page=delcampe-license' ) ),
        esc_html( $license_text )
    );
    
    array_unshift( $links, $settings_link, $license_link );
    
    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'delcampe_plugin_action_links' );

/**
 * Add plugin meta links
 * Adds Documentation and Support links to the plugin listing
 * 
 * @since  1.6.2.1
 * @param  array  $links Existing plugin meta links
 * @param  string $file  Plugin basename
 * @return array  Modified meta links
 */
function delcampe_plugin_row_meta( $links, $file ) {
    if ( plugin_basename( __FILE__ ) === $file ) {
        // Use hardcoded strings if text domain is not yet loaded
        $check_updates_text = 'Check for updates';
        $docs_text = 'Documentation';
        $support_text = 'Support';
        
        // Check if text domain is loaded, then translate
        if ( is_textdomain_loaded( 'wc-delcampe-integration' ) ) {
            $check_updates_text = __( 'Check for updates', 'wc-delcampe-integration' );
            $docs_text = __( 'Documentation', 'wc-delcampe-integration' );
            $support_text = __( 'Support', 'wc-delcampe-integration' );
        }
        
        $row_meta = array();
        
        // Add "Check for updates" link that triggers our custom update check
        $row_meta['check_updates'] = sprintf(
            '<a href="%s">%s</a>',
            esc_url( wp_nonce_url( 
                admin_url( 'admin.php?delcampe_check_updates=1' ), 
                'delcampe_check_updates' 
            ) ),
            esc_html( $check_updates_text )
        );
        
        $row_meta['docs'] = sprintf(
            '<a href="%s" target="_blank">%s</a>',
            esc_url( 'https://www.sos-tech.ca/docs/delcampe-sync' ),
            esc_html( $docs_text )
        );
        
        $row_meta['support'] = sprintf(
            '<a href="%s" target="_blank">%s</a>',
            esc_url( 'https://www.sos-tech.ca/support/delcampe-sync' ),
            esc_html( $support_text )
        );
        
        return array_merge( $links, $row_meta );
    }
    
    return $links;
}
add_filter( 'plugin_row_meta', 'delcampe_plugin_row_meta', 10, 2 );
